package com.atlassian.cache.ehcache;

import java.util.SortedMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.cache.CacheException;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.CachedReferenceListener;
import com.atlassian.cache.impl.CachedReferenceListenerSupport;
import com.atlassian.cache.impl.LazyCachedReferenceListenerSupport;
import com.atlassian.cache.impl.ReferenceKey;
import com.atlassian.util.concurrent.Supplier;

import com.google.common.collect.ImmutableSortedMap;

import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.event.CacheEventListener;

import static com.atlassian.cache.ehcache.DelegatingCacheStatistics.toStatistics;

/**
 * A Lazy Reference that delegates to EhCache.
 *
 * @since 2.0
 */
class DelegatingCachedReference<V> extends ManagedCacheSupport implements CachedReference<V>
{
    private final Ehcache delegate;
    private final CachedReferenceListenerSupport<V> listenerSupport = new LazyCachedReferenceListenerSupport<V>()
    {
        @Override
        protected void init()
        {
            delegate.getCacheEventNotificationService().registerListener(new DelegatingReferenceCacheEventListener());
        }
    };

    private DelegatingCachedReference(final Ehcache delegate, CacheSettings settings)
    {
        super(delegate, settings);
        this.delegate = delegate;
    }

    static <V> DelegatingCachedReference<V> create(final Ehcache delegate, CacheSettings settings)
    {
        return new DelegatingCachedReference<V>(delegate, settings);
    }

    @Nonnull
    @SuppressWarnings("unchecked")
    @Override
    public V get()
    {
        try
        {
            return (V)delegate.get(ReferenceKey.KEY).getObjectValue();
        }
        catch (net.sf.ehcache.CacheException e)
        {
            throw new CacheException(e.getCause());
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void reset()
    {
        try
        {
            delegate.remove(ReferenceKey.KEY);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void clear()
    {
        reset();
    }

    public boolean equals(@Nullable final Object other)
    {
        if (other instanceof DelegatingCachedReference)
        {
            DelegatingCachedReference<?> otherDelegatingReference = (DelegatingCachedReference<?>) other;
            if (delegate.equals(otherDelegatingReference.delegate))
            {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode()
    {
        return 3 + delegate.hashCode();
    }

    @Nonnull
    @Override
    public SortedMap<CacheStatisticsKey,Supplier<Long>> getStatistics()
    {
        if (isStatisticsEnabled())
        {
            return toStatistics(delegate.getStatistics());
        }
        else
        {
            return ImmutableSortedMap.of();
        }
    }

    @Override
    public void addListener(@Nonnull CachedReferenceListener<V> listener, boolean includeValues)
    {
        listenerSupport.add(listener, includeValues);
    }

    @Override
    public void removeListener(@Nonnull CachedReferenceListener<V> listener)
    {
        listenerSupport.remove(listener);
    }

    private class DelegatingReferenceCacheEventListener implements CacheEventListener
    {
        @Override
        public void notifyElementRemoved(Ehcache ehcache, Element element) throws net.sf.ehcache.CacheException
        {
            listenerSupport.notifyReset((V) element.getObjectValue());
        }

        @Override
        public void notifyElementPut(Ehcache ehcache, Element element) throws net.sf.ehcache.CacheException
        {
            listenerSupport.notifySet((V) element.getObjectValue());
        }

        @Override
        public void notifyElementUpdated(Ehcache ehcache, Element element) throws net.sf.ehcache.CacheException
        {
            listenerSupport.notifySet((V) element.getObjectValue());
        }

        @Override
        public void notifyElementExpired(Ehcache ehcache, Element element)
        {
            listenerSupport.notifyEvict((V) element.getObjectValue());
        }

        @Override
        public void notifyElementEvicted(Ehcache ehcache, Element element)
        {
            listenerSupport.notifyEvict((V) element.getObjectValue());
        }

        @Override
        public void notifyRemoveAll(Ehcache ehcache)
        {
            listenerSupport.notifyReset((V) null);
        }

        @Override
        public void dispose()
        {
            // We don't hold onto any resources so there is nothing to be done.
        }

        public Object clone() throws CloneNotSupportedException
        {
            throw new CloneNotSupportedException();
        }
    }
}
