package com.atlassian.cache.ehcache;

import java.util.Collection;
import java.util.SortedMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheEntryListener;
import com.atlassian.cache.CacheException;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.cache.impl.CacheEntryListenerSupport;
import com.atlassian.cache.impl.LazyCacheEntryListenerSupport;
import com.atlassian.util.concurrent.Supplier;

import com.google.common.collect.ImmutableSortedMap;

import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.event.CacheEventListener;

import static com.atlassian.cache.ehcache.DelegatingCacheStatistics.toStatistics;

/**
 * A Cache that delegates to EhCache.
 *
 * @since 2.0
 */
class DelegatingCache<K, V> extends ManagedCacheSupport implements Cache<K, V>
{
    private final Ehcache delegate;

    private final CacheEntryListenerSupport<K, V> listenerSupport = new LazyCacheEntryListenerSupport<K, V>()
    {
        @Override
        protected void init()
        {
            delegate.getCacheEventNotificationService().registerListener(new DelegatingCacheEventListener());
        }
    };

    private DelegatingCache(final Ehcache delegate, CacheSettings settings)
    {
        super(delegate, settings);
        this.delegate = delegate;
    }

    static <K, V> DelegatingCache<K, V> create(final Ehcache delegate, CacheSettings settings)
    {
        return new DelegatingCache<K, V>(delegate, settings);
    }

    @Override
    public boolean containsKey(@Nonnull K key)
    {
        return delegate.isKeyInCache(key);
    }

    @Nonnull
    @SuppressWarnings("unchecked")
    @Override
    public Collection<K> getKeys()
    {
        try
        {
            return delegate.getKeys();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void put(@Nonnull final K key, @Nonnull final V value)
    {
        try
        {
            delegate.put(new Element(key, value));
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @SuppressWarnings("unchecked")
    @Nullable
    @Override
    public V get(@Nonnull final K key)
    {
        try
        {
            Element element = delegate.get(key);
            return element == null ? null : (V) element.getObjectValue();
        }
        catch (net.sf.ehcache.CacheException e)
        {
            throw new CacheException(e.getCause());
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @SuppressWarnings("unchecked")
    @Nonnull
    @Override
    public V get(@Nonnull final K key, @Nonnull final com.atlassian.cache.Supplier<? extends V> valueSupplier)
    {
        try
        {
            Element element = delegate.getWithLoader(key, new ReferenceCacheLoader(valueSupplier), null);
            return (V) element.getObjectValue();
        }
        catch (net.sf.ehcache.CacheException e)
        {
            throw new CacheException(e.getCause());
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void remove(@Nonnull final K key)
    {
        try
        {
            delegate.remove(key);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void removeAll()
    {
        try
        {
            delegate.removeAll();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void clear()
    {
        removeAll();
    }

    @Nullable
    @Override
    @SuppressWarnings("unchecked")
    public V putIfAbsent(@Nonnull K key, @Nonnull V value)
    {
        try
        {
            Element previous = delegate.putIfAbsent(new Element(key, value));
            if (previous != null)
            {
                return (V) previous.getObjectValue();
            }
            else
            {
                return null;
            }
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public boolean remove(@Nonnull K key, @Nonnull V value)
    {
        try
        {
            return delegate.removeElement(new Element(key, value));
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public boolean replace(@Nonnull K key, @Nonnull V oldValue, @Nonnull V newValue)
    {
        try
        {
            return delegate.replace(new Element(key, oldValue), new Element(key, newValue));
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Nonnull
    @Override
    public SortedMap<CacheStatisticsKey,Supplier<Long>> getStatistics()
    {
        if (isStatisticsEnabled())
        {
            return toStatistics(delegate.getStatistics());
        }
        else
        {
            return ImmutableSortedMap.of();
        }
    }

    @Override
    public boolean equals(@Nullable final Object other)
    {
        if (other instanceof DelegatingCache)
        {
            DelegatingCache<?,?> otherDelegatingCache = (DelegatingCache<?,?>)other;
            if (delegate.equals(otherDelegatingCache.delegate))
            {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode()
    {
        return 3 + delegate.hashCode();
    }

    @Override
    public void addListener(@Nonnull CacheEntryListener<K, V> listener, boolean includeValues)
    {
        listenerSupport.add(listener, includeValues);
    }

    @Override
    public void removeListener(@Nonnull CacheEntryListener<K, V> listener)
    {
        listenerSupport.remove(listener);
    }

    private class DelegatingCacheEventListener implements CacheEventListener
    {
        @Override
        public void notifyElementRemoved(Ehcache ehcache, Element element) throws net.sf.ehcache.CacheException
        {
            listenerSupport.notifyRemove((K) element.getObjectKey(), (V) element.getObjectValue());
        }

        @Override
        public void notifyElementPut(Ehcache ehcache, Element element) throws net.sf.ehcache.CacheException
        {
            listenerSupport.notifyAdd((K) element.getObjectKey(), (V) element.getObjectValue());
        }

        @Override
        public void notifyElementUpdated(Ehcache ehcache, Element element) throws net.sf.ehcache.CacheException
        {
            listenerSupport.notifyUpdate((K) element.getObjectKey(), (V) element.getObjectValue(), null);
        }

        @Override
        public void notifyElementExpired(Ehcache ehcache, Element element)
        {
            listenerSupport.notifyEvict((K) element.getObjectKey(), (V) element.getObjectValue());
        }

        @Override
        public void notifyElementEvicted(Ehcache ehcache, Element element)
        {
            listenerSupport.notifyEvict((K) element.getObjectKey(), (V) element.getObjectValue());
        }

        @Override
        public void notifyRemoveAll(Ehcache ehcache)
        {
            // There is no way to enumerate the keys that were in the cache, therefore we cannot
            // produce any meaningful event
        }

        @Override
        public void dispose()
        {
            // We don't hold onto any resources so there is nothing to be done.
        }

        public Object clone() throws CloneNotSupportedException
        {
            throw new CloneNotSupportedException();
        }
    }
}
