package com.atlassian.cache.compat.delegate;

import java.util.concurrent.TimeUnit;

import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @since v1.0
 */
public class CacheSettingsMapper
{
    static final Logger LOG = LoggerFactory.getLogger(CacheSettingsMapper.class);

    private CacheSettingsMapper()
    {
        // static-only class
    }

    static CacheSettings mapCacheSettings(com.atlassian.cache.compat.CacheSettings compatCacheSettings)
    {
        if (compatCacheSettings == null)
        {
            return null;
        }

        final CacheSettingsBuilder builder = new CacheSettingsBuilder();
        applyExpireAfterAccess(builder, compatCacheSettings.getExpireAfterAccess());
        applyExpireAfterWrite(builder, compatCacheSettings.getExpireAfterWrite());
        applyFlushable(builder, compatCacheSettings.getFlushable());
        applyLocal(builder, compatCacheSettings.getLocal());
        applyMaxEntries(builder, compatCacheSettings.getMaxEntries());
        applyReplicateAsynchronously(builder, compatCacheSettings.getReplicateAsynchronously());
        applyReplicateViaCopy(builder, compatCacheSettings.getReplicateViaCopy());
        return builder.build();
    }

    // As an oversight, CacheSettings.replicateViaInvalidation() and CacheSettings.replicateSynchronously() are not
    // available in atlassian-cache until v2.3, so we are not guaranteed to find them.
    private static void applyUsingReflection(CacheSettingsBuilder builder, String methodName)
    {
        try
        {
            builder.getClass().getMethod(methodName).invoke(builder);
        }
        catch (Exception e)
        {
            if (LOG.isDebugEnabled())
            {
                LOG.warn("Unable to apply explicit {} setting.", methodName, e);
            }
            else
            {
                LOG.warn("Unable to apply explicit {} setting.  Enable DEBUG logging for a stack trace.", methodName);
            }
        }
    }

    private static void applyReplicateViaCopy(CacheSettingsBuilder builder, Boolean replicateViaCopy)
    {
        if (replicateViaCopy != null)
        {
            if (replicateViaCopy)
            {
                builder.replicateViaCopy();
            }
            else
            {
                applyUsingReflection(builder, "replicateViaInvalidation");
            }
        }
    }

    private static void applyReplicateAsynchronously(CacheSettingsBuilder builder, Boolean replicateAsync)
    {
        if (replicateAsync != null)
        {
            if (replicateAsync)
            {
                builder.replicateAsynchronously();
            }
            else
            {
                applyUsingReflection(builder, "replicateSynchronously");
            }
        }
    }

    private static void applyMaxEntries(CacheSettingsBuilder builder, Integer maxEntries)
    {
        if (maxEntries != null)
        {
            builder.maxEntries(maxEntries);
        }
    }

    private static void applyLocal(CacheSettingsBuilder builder, Boolean local)
    {
        if (local != null)
        {
            if (local)
            {
                builder.local();
            }
            else
            {
                builder.remote();
            }
        }
    }

    private static void applyFlushable(CacheSettingsBuilder builder, Boolean flushable)
    {
        if (flushable != null)
        {
            if (flushable)
            {
                builder.flushable();
            }
            else
            {
                builder.unflushable();
            }
        }
    }

    private static void applyExpireAfterWrite(CacheSettingsBuilder builder, Long expireAfterWrite)
    {
        if (expireAfterWrite != null)
        {
            builder.expireAfterWrite(expireAfterWrite, TimeUnit.MILLISECONDS);
        }
    }

    private static void applyExpireAfterAccess(CacheSettingsBuilder builder, Long expireAfterAccess)
    {
        if (expireAfterAccess != null)
        {
            builder.expireAfterAccess(expireAfterAccess, TimeUnit.MILLISECONDS);
        }
    }

}
