package com.atlassian.cache.impl;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import com.atlassian.cache.CachedReferenceListener;

import static java.util.Objects.requireNonNull;


/**
 * A utility class which:
 * <ul>
 *     <li>Manages a set of {@code CachedReferenceListener}s</li>
 *     <li>Provides a way to notify these listeners about different events</li>
 * </ul>
 */
public class DefaultCachedReferenceListenerSupport<V> implements CachedReferenceListenerSupport<V>
{
    private final Set<CachedReferenceListener<V>> listeners = new CopyOnWriteArraySet<CachedReferenceListener<V>>();
    private final CachedReferenceNotificationSupport notificationSupport =
            CachedReferenceNotificationSupport.getInstance();

    @Override
    public void add(CachedReferenceListener<V> listener, boolean includeValues)
    {
        listeners.add(requireNonNull(listener));
    }

    @Override
    public void remove(CachedReferenceListener<V> listener)
    {
        listeners.remove(requireNonNull(listener));
    }

    @Override
    public void notifyEvict(V oldValue)
    {
        notificationSupport.notifyEvict(listeners, new DefaultCachedReferenceEvent<V>(oldValue));
    }

    @Override
    public void notifySet(V value)
    {
        notificationSupport.notifySet(listeners, new DefaultCachedReferenceEvent<V>(value));
    }

    @Override
    public void notifyReset(V oldValue)
    {
        notificationSupport.notifyReset(listeners, new DefaultCachedReferenceEvent<V>(oldValue));
    }
}
