package com.atlassian.cache.impl;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import com.atlassian.cache.CacheEntryListener;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A utility class which:
 * <ul>
 *     <li>Manages a set of {@code CacheEntryListener}s</li>
 *     <li>Provides a way to notify these listeners about different events</li>
 * </ul>
 */
public class DefaultCacheEntryListenerSupport<K, V> implements CacheEntryListenerSupport<K, V>
{
    private final Set<CacheEntryListener<K, V>> listeners = new CopyOnWriteArraySet<CacheEntryListener<K, V>>();
    private final CacheEntryNotificationSupport notificationSupport = CacheEntryNotificationSupport.getInstance();

    public void add(CacheEntryListener<K, V> listener, boolean includeValues)
    {
        listeners.add(checkNotNull(listener));
    }

    public void remove(CacheEntryListener<K, V> listener)
    {
        listeners.remove(checkNotNull(listener));
    }

    public void notifyAdd(K key, V value)
    {
        notificationSupport.notifyAdd(listeners, new DefaultCacheEntryEvent<K, V>(key, value, null));
    }

    public void notifyEvict(K key, V oldValue)
    {
        notificationSupport.notifyEvict(listeners, new DefaultCacheEntryEvent<K, V>(key, null, oldValue));
    }

    public void notifyRemove(K key, V oldValue)
    {
        notificationSupport.notifyRemove(listeners, new DefaultCacheEntryEvent<K, V>(key, null, oldValue));
    }

    public void notifyUpdate(K key, V value, V oldValue)
    {
        notificationSupport.notifyUpdate(listeners, new DefaultCacheEntryEvent<K, V>(key, value, oldValue));
    }
}
