package com.atlassian.braid.transformation;

import com.atlassian.braid.Extension;
import com.atlassian.braid.FieldRename;
import com.atlassian.braid.Link;
import com.atlassian.braid.SchemaNamespace;
import com.atlassian.braid.SchemaSource;
import com.atlassian.braid.TypeRename;
import graphql.language.FieldDefinition;
import graphql.language.InputValueDefinition;
import graphql.language.ListType;
import graphql.language.NonNullType;
import graphql.language.ObjectTypeDefinition;
import graphql.language.Type;
import graphql.language.TypeDefinition;
import graphql.language.TypeName;
import graphql.schema.idl.TypeDefinitionRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import static com.atlassian.braid.TypeUtils.DEFAULT_QUERY_TYPE_NAME;
import static com.atlassian.braid.TypeUtils.findMutationType;
import static com.atlassian.braid.TypeUtils.findQueryType;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;

/**
 * This wraps a {@link SchemaSource} to enhance it with helper functions
 */
public final class BraidSchemaSource {
    private static final Logger log = LoggerFactory.getLogger(BraidSchemaSource.class);

    private final SchemaSource schemaSource;

    private final TypeDefinitionRegistry registry;

    public BraidSchemaSource(SchemaSource schemaSource) {
        this.schemaSource = requireNonNull(schemaSource);
        this.registry = schemaSource.getSchema();
    }

    public SchemaSource getSchemaSource() {
        return schemaSource;
    }

    public SchemaNamespace getNamespace() {
        return schemaSource.getNamespace();
    }

    List<Extension> getExtensions(String type) {
        return schemaSource.getExtensions().stream().filter(e -> e.getType().equals(type)).collect(toList());
    }

    Optional<TypeRename> getTypeRenameFromSourceName(String type) {
        return schemaSource.getTypeRenames().stream().filter(a -> a.getSourceName().equals(type)).findFirst();
    }

    public Optional<TypeRename> getTypeRenameFromBraidName(String type) {
        return schemaSource.getTypeRenames().stream().filter(a -> a.getBraidName().equals(type)).findFirst();
    }

    public String getBraidTypeName(String sourceTypeName) {
        return getTypeRenameFromSourceName(sourceTypeName)
                .map(TypeRename::getBraidName)
                .orElse(sourceTypeName);
    }

    public String getSourceTypeName(String braidTypeName) {
        return getTypeRenameFromBraidName(braidTypeName)
                .map(TypeRename::getSourceName)
                .orElse(braidTypeName);
    }

    Optional<FieldRename> getQueryFieldRenames(String sourceFieldName) {
        return getFieldRenameBySourceName(schemaSource.getQueryFieldRenames(), sourceFieldName);
    }

    Optional<FieldRename> getMutationFieldRenames(String sourceFieldName) {
        return getFieldRenameBySourceName(schemaSource.getMutationFieldRenames(), sourceFieldName);
    }

    public String getOperationNamePrefix() { return schemaSource.getOperationNamePrefix(); }

    /**
     * Gets the actual source type, accounting for links to query objects that have been renamed when merged into the
     * Braid schema
     */
    public String getLinkBraidSourceType(Link link) {
        return getQueryType()
                .flatMap(maybeGetQueryTypeNameIfLinkSourceIsQueryType(link))
                .orElseGet(link::getSourceType);
    }

    private Function<ObjectTypeDefinition, Optional<String>> maybeGetQueryTypeNameIfLinkSourceIsQueryType(Link link) {
        return originalQueryType ->
                isLinkSourceTypeQueryType(link, originalQueryType) ? Optional.of(DEFAULT_QUERY_TYPE_NAME) : empty();
    }

    private boolean isLinkSourceTypeQueryType(Link link, ObjectTypeDefinition originalQueryType) {
        return originalQueryType.getName().equals(link.getSourceType());
    }

    public Collection<BraidTypeDefinition> getNonOperationTypes() {
        ObjectTypeDefinition queryType = findQueryType(registry).orElse(null);
        ObjectTypeDefinition mutationType = findMutationType(registry).orElse(null);
        return registry.types()
                .values()
                .stream()
                .filter(type -> type != queryType && type != mutationType)
                .map(td -> new BraidTypeDefinition(this, td))
                .collect(toList());
    }

    boolean hasType(String type) {
        return getType(type).isPresent();
    }

    Optional<TypeDefinition> getType(String type) {
        return registry.getType(type);
    }

    Optional<ObjectTypeDefinition> getQueryType() {
        return findQueryType(registry);
    }

    Optional<ObjectTypeDefinition> getMutationType() {
        return findMutationType(registry);
    }

    public TypeDefinitionRegistry getTypeRegistry() {
        return registry;
    }

    Type renameTypeToBraidName(Type type) {
        if (type instanceof TypeName) {
            final String typeName = ((TypeName) type).getName();
            TypeRename typeRename = getTypeRenameFromSourceName(typeName).orElse(TypeRename.from(typeName, typeName));
            return new TypeName(typeRename.getBraidName());
        } else if (type instanceof NonNullType) {
            return new NonNullType(renameTypeToBraidName(((NonNullType) type).getType()));
        } else if (type instanceof ListType) {
            return new ListType(renameTypeToBraidName(((ListType) type).getType()));
        } else {
            // TODO handle all definition types (in a generic enough manner)
            log.error("Definition type : " + type + " not handled correctly for aliases.  Please raise an issue.");
            return type;
        }
    }

    List<InputValueDefinition> renameInputValueDefinitionsToBraidTypes(List<InputValueDefinition> inputValueDefinitions) {
        return inputValueDefinitions.stream()
                .map(input ->
                        InputValueDefinition.newInputValueDefinition()
                                .name(input.getName())
                                .type(renameTypeToBraidName(input.getType()))
                                .defaultValue(input.getDefaultValue())
                                .directives(input.getDirectives())
                                .build())
                .collect(toList());
    }

    private Optional<FieldRename> getFieldRenameBySourceName(List<FieldRename> fieldRenames, String sourceName) {
        return fieldRenames.stream().filter(a -> a.getSourceName().equals(sourceName)).findFirst();
    }

    private Optional<FieldRename> getFieldRenameByBraidName(List<FieldRename> fieldRenames, String braidFieldName) {
        return fieldRenames.stream().filter(a -> a.getBraidName().equals(braidFieldName)).findFirst();
    }

    boolean hasTypeAndField(TypeDefinitionRegistry registry, TypeDefinition typeDef, FieldDefinition fieldDef) {
        ObjectTypeDefinition queryType = findQueryType(registry).orElse(null);

        if (queryType != null && queryType == typeDef) {
            if (schemaSource.getQueryFieldRenames().isEmpty()) {
                return queryType.getFieldDefinitions().stream()
                        .anyMatch(fieldDefinition -> fieldDefinition.getName().equals(fieldDef.getName()));
            } else {
                return getFieldRenameByBraidName(schemaSource.getQueryFieldRenames(), fieldDef.getName())
                        .map(alias -> alias.getBraidName().equals(fieldDef.getName()))
                        .orElse(false);
            }
        } else {
            return getType(getSourceTypeName(typeDef.getName())).isPresent();
        }
    }
}
