package com.atlassian.braid;

import com.atlassian.braid.source.QueryExecutorSchemaSource;
import graphql.language.Document;
import graphql.language.Field;
import graphql.language.FragmentDefinition;
import graphql.language.OperationDefinition;
import graphql.language.Selection;
import graphql.language.SelectionSet;
import graphql.language.VariableDefinition;
import graphql.schema.DataFetchingEnvironment;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;


/**
 * The context of a field in graphql request. This context is passed on while applying <code>FieldTransformation</code>s,
 * during which the selections, variable definitions, and fragment definitions are accumulated. Once the transformations
 * are completed, a <code>Document</code> for querying is available in the context.
 */
public class FieldTransformationContext {

    private final Map<String, Object> variables;
    private final Map<DataFetchingEnvironment, List<FieldKey>> clonedFields;
    private final AtomicInteger counter;
    private final Map<FieldKey, Object> shortCircuitedData;
    private final QueryExecutorSchemaSource schemaSource;
    private final String operationName;
    private final OperationDefinition.Operation operationType;
    private final List<Selection<?>> selections;
    private final List<VariableDefinition> variableDefinitions;
    private final List<FragmentDefinition> fragmentDefinitions;

    private final Set<String> existingVariableDefinitionNames;
    //this keeps track of the fragmentDefinition names to avoid adding duplicate fragments to ensure uniqueness in fragment names
    private final Set<String> existingFragmentDefinitionNames;
    private final List<Field> missingFields = new ArrayList<>();

    public FieldTransformationContext(QueryExecutorSchemaSource schemaSource, String operationName, OperationDefinition.Operation operationType) {
        this.schemaSource = schemaSource;
        this.operationName = operationName;
        this.operationType = operationType;
        this.selections = new ArrayList<>();
        this.variableDefinitions = new ArrayList<>();
        this.fragmentDefinitions = new ArrayList<>();

        Document document = null;

        variables = new HashMap<>();
        clonedFields = new HashMap<>();
        existingVariableDefinitionNames = new HashSet<>();
        existingFragmentDefinitionNames = new HashSet<>();

        // start at 99 so that we can find variables already counter-namespaced via startsWith()
        counter = new AtomicInteger(99);

        // this is to gather data we don't need to fetch through batch loaders, e.g. when on the the variable used in
        // the query is fetched
        shortCircuitedData = new HashMap<>();
    }

    public void addSelection(Selection<?> selection) {
        selections.add(selection);
    }

    public Map<String, Object> getVariables() {
        return variables;
    }

    public OperationDefinition getOperation() {
        return OperationDefinition.newOperationDefinition()
                .name(operationName)
                .operation(operationType)
                .selectionSet(SelectionSet.newSelectionSet()
                        .selections(selections)
                        .build())
                .variableDefinitions(variableDefinitions)
                .build();
    }

    public Document getDocument() {
        Document.Builder documentBuilder = Document.newDocument();
        documentBuilder.definition(getOperation());
        for (FragmentDefinition fragmentDefinition: fragmentDefinitions) {
            documentBuilder.definition(fragmentDefinition);
        }
        return documentBuilder.build();
    }

    public void addVariableDefinition(VariableDefinition variableDefinition) {
        if (!existingVariableDefinitionNames.contains(variableDefinition.getName())) {
            variableDefinitions.add(variableDefinition);
            existingVariableDefinitionNames.add(variableDefinition.getName());
        }
    }

    public void addFragmentDefinition(FragmentDefinition definition) {
        // Fragment is added only if it doesn't exist
        if (!existingFragmentDefinitionNames.contains(definition.getName())) {
            fragmentDefinitions.add(definition);
            existingFragmentDefinitionNames.add(definition.getName());
        }
    }

    public Map<DataFetchingEnvironment, List<FieldKey>> getClonedFields() {
        return clonedFields;
    }

    public AtomicInteger getCounter() {
        return counter;
    }

    public QueryExecutorSchemaSource getSchemaSource() {
        return schemaSource;
    }

    public List<Field> getMissingFields() {
        return missingFields;
    }

    public Map<FieldKey, Object> getShortCircuitedData() {
        return shortCircuitedData;
    }

    public void addMissingFields(List<Field> missingFields) {
        this.missingFields.clear();
        this.missingFields.addAll(missingFields);
    }
}
