package com.atlassian.braid;

import graphql.schema.DataFetchingEnvironment;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static java.util.concurrent.CompletableFuture.completedFuture;

public class DefaultArgumentValueProvider implements ArgumentValueProvider {
    public static final ArgumentValueProvider INSTANCE = new DefaultArgumentValueProvider();

    private final Map<String, Object> staticValues;

    private DefaultArgumentValueProvider() {
        this.staticValues = Collections.emptyMap();
    }

    public DefaultArgumentValueProvider(@Nullable Map<String, Object> staticValues) {
        this.staticValues = Optional.ofNullable(staticValues)
                .map(Collections::unmodifiableMap)
                .orElse(Collections.emptyMap());
    }

    @Override
    public CompletableFuture<Object> fetchValueForArgument(LinkArgument linkArgument, DataFetchingEnvironment environment) {
        switch (linkArgument.getArgumentSource()) {
            case OBJECT_FIELD:
                return BatchLoaderUtils.getFieldValue(environment, linkArgument.getSourceName())
                        .thenApply(v -> v.orElse(null));
            case CONTEXT:
                throw new UnsupportedOperationException("not supported in default implementation");
            case FIELD_ARGUMENT:
                return completedFuture(environment.getArgument(linkArgument.getSourceName()));
            case STATIC:
                return completedFuture(staticValues.get(linkArgument.getSourceName()));
            default:
                throw new UnsupportedOperationException("Unsupported argument source.");
        }
    }
}
