package com.atlassian.braid;

import graphql.execution.DataFetcherResult;
import graphql.language.Field;

import java.util.Objects;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * Links a field on a type to another data source
 */
@SuppressWarnings("WeakerAccess")
public final class Link {

    private final SchemaNamespace sourceNamespace;
    private final String sourceType;

    private final SchemaNamespace targetNamespace;
    private final String targetType;
    private final String newFieldName;

    private final String topLevelQueryField;
    private final String queryArgumentName;
    private final String sourceInputFieldName;
    private final String targetFieldMatchingQueryArgument;
    private final boolean removeInputField;


    /**
     * Whether a null source field value should prompt a remote link call
     */
    private final boolean nullable;

    private final boolean noSchemaChangeNeeded;

    public interface CustomTransformation {
        void createQuery(Field field, Object targetId);

        DataFetcherResult<Object> unapplyForResult(Field field, DataFetcherResult<Object> dataFetcherResult);
    }

    private final CustomTransformation customTransformation;

    public Link(SchemaNamespace sourceNamespace,
                String sourceType,
                SchemaNamespace targetNamespace,
                String targetType,
                String newFieldName,
                String topLevelQueryField,
                String queryArgumentName,
                String sourceInputFieldName,
                String targetFieldMatchingQueryArgument,
                boolean removeInputField,
                boolean nullable,
                boolean noSchemaChangeNeeded,
                CustomTransformation customTransformation) {
        this.sourceNamespace = requireNonNull(sourceNamespace);
        this.sourceType = requireNonNull(sourceType);
        this.targetNamespace = requireNonNull(targetNamespace);
        this.targetType = requireNonNull(targetType);
        this.newFieldName = requireNonNull(newFieldName);
        this.topLevelQueryField = requireNonNull(topLevelQueryField);
        this.queryArgumentName = requireNonNull(queryArgumentName);
        this.sourceInputFieldName = requireNonNull(sourceInputFieldName);
        this.targetFieldMatchingQueryArgument = requireNonNull(targetFieldMatchingQueryArgument);
        this.removeInputField = removeInputField;
        this.nullable = nullable;
        this.noSchemaChangeNeeded = noSchemaChangeNeeded;
        this.customTransformation = customTransformation;
    }


    /**
     * @deprecated use .newLink() Builder
     */
    @Deprecated
    public static LinkBuilder from(SchemaNamespace sourceNamespace, String sourceType, String newFieldName) {
        return newLink()
                .sourceNamespace(sourceNamespace)
                .sourceType(sourceType)
                .newFieldName(newFieldName);
    }

    /**
     * @deprecated use .newLink() Builder
     */
    @Deprecated
    public static LinkBuilder from(SchemaNamespace sourceNamespace, String sourceType, String newFieldName, String sourceInputFieldName) {
        return newLink()
                .sourceNamespace(sourceNamespace)
                .sourceType(sourceType)
                .newFieldName(newFieldName)
                .sourceInputFieldName(sourceInputFieldName);
    }

    public String getSourceType() {
        return sourceType;
    }

    /**
     * @return the field name within the {@link #getSourceType() source type} that the link creates
     *
     * @deprecated use {@link #getNewFieldName()}
     */
    @Deprecated
    public String getSourceField() {
        return newFieldName;
    }

    /**
     * @return the field name within the {@link #getSourceType() source type} that is used to query the linked 'object'
     *
     * @deprecated use {@link #getSourceInputFieldName()}
     */
    @Deprecated
    public String getSourceFromField() {
        return sourceInputFieldName;
    }

    /**
     * @return whether the {@link #getSourceFromField()} should be removed from the final schema, ie. no longer appear
     * as a separate, standalone field within the {@link #getSourceType() source type}
     *
     * @deprecated use {@link #isRemoveInputField()}
     */
    @Deprecated
    public boolean isReplaceFromField() {
        return removeInputField;
    }

    /**
     * @return the namespace of the schema where the target 'object' should be queried
     */
    public SchemaNamespace getTargetNamespace() {
        return targetNamespace;
    }

    /**
     * @return the type of the target field to which the link exists
     * (e.g the output type of the query to the target schema)
     */
    public String getTargetType() {
        return targetType;
    }

    /**
     * @return the name of the query field used to retrieve the linked object.
     *
     * @deprecated use {@link #getTopLevelQueryField()}
     */
    @Deprecated
    public String getTargetQueryField() {
        return topLevelQueryField;
    }

    /**
     * The name of the field in the target object that corresponds to the field used in the query variables.
     * <p>This allows for example to by pass querying for said field by directly outputting the passed value.
     *
     * @return the name of the field in the target object.
     *
     * @deprecated use {@link #getTargetFieldMatchingQueryArgument()}
     */
    @Deprecated
    public String getTargetVariableQueryField() {
        return targetFieldMatchingQueryArgument;
    }

    /**
     * @return the name of the query argument used to retrieve the linked object. This argument will be given the value
     * denoted by the {@link #getSourceFromField() source from field}
     *
     * @deprecated use {@link #getQueryArgumentName()}
     */
    @Deprecated
    public String getArgumentName() {
        return queryArgumentName;
    }

    public boolean isNullable() {
        return nullable;
    }

    public SchemaNamespace getSourceNamespace() {
        return sourceNamespace;
    }

    public String getNewFieldName() {
        return newFieldName;
    }

    public String getTopLevelQueryField() {
        return topLevelQueryField;
    }

    public String getQueryArgumentName() {
        return queryArgumentName;
    }

    public String getSourceInputFieldName() {
        return sourceInputFieldName;
    }

    public String getTargetFieldMatchingQueryArgument() {
        return targetFieldMatchingQueryArgument;
    }

    public boolean isRemoveInputField() {
        return removeInputField;
    }

    public boolean isNoSchemaChangeNeeded() {
        return noSchemaChangeNeeded;
    }

    public CustomTransformation getCustomTransformation() {
        return customTransformation;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Link link = (Link) o;
        return removeInputField == link.removeInputField &&
                nullable == link.nullable &&
                Objects.equals(sourceNamespace, link.sourceNamespace) &&
                Objects.equals(sourceType, link.sourceType) &&
                Objects.equals(targetNamespace, link.targetNamespace) &&
                Objects.equals(targetType, link.targetType) &&
                Objects.equals(newFieldName, link.newFieldName) &&
                Objects.equals(topLevelQueryField, link.topLevelQueryField) &&
                Objects.equals(queryArgumentName, link.queryArgumentName) &&
                Objects.equals(sourceInputFieldName, link.sourceInputFieldName) &&
                Objects.equals(targetFieldMatchingQueryArgument, link.targetFieldMatchingQueryArgument);
    }

    @Override
    public int hashCode() {
        return Objects.hash(sourceNamespace, sourceType, targetNamespace, targetType, newFieldName, topLevelQueryField, queryArgumentName, sourceInputFieldName, targetFieldMatchingQueryArgument, removeInputField, nullable);
    }

    @Override
    public String toString() {
        return "Link{" +
                "sourceNamespace=" + sourceNamespace +
                ", sourceType='" + sourceType + '\'' +
                ", targetNamespace=" + targetNamespace +
                ", targetType='" + targetType + '\'' +
                ", newFieldName='" + newFieldName + '\'' +
                ", topLevelQueryField='" + topLevelQueryField + '\'' +
                ", queryArgumentName='" + queryArgumentName + '\'' +
                ", sourceInputFieldName='" + sourceInputFieldName + '\'' +
                ", targetFieldMatchingQueryArgument='" + targetFieldMatchingQueryArgument + '\'' +
                ", replaceInputField=" + removeInputField +
                ", nullable=" + nullable +
                '}';
    }

    public static LinkBuilder newLink() {
        return new LinkBuilder();
    }

    public static final class LinkBuilder {

        private SchemaNamespace targetNamespace;
        private String targetType;

        private String topLevelQueryField;
        private String queryArgumentName;
        private String sourceInputFieldName;

        private String targetFieldMatchingQueryArgument;

        private boolean removeInputField;

        private SchemaNamespace sourceNamespace;
        private String sourceType;

        private String newFieldName;
        private boolean nullable;

        private boolean noSchemeChangeNeeded;
        private CustomTransformation customTransformation;

        private LinkBuilder() {
        }

        /**
         * @deprecated use single methods
         */
        @Deprecated
        public LinkBuilder to(SchemaNamespace targetNamespace, String targetType) {
            return to(targetNamespace, targetType, null);
        }

        /**
         * @deprecated use single methods
         */
        @Deprecated
        public LinkBuilder to(SchemaNamespace targetNamespace, String targetType, String topLevelQueryField) {
            return to(targetNamespace, targetType, topLevelQueryField, null);
        }

        /**
         * @deprecated use single methods
         */
        @Deprecated
        public LinkBuilder to(SchemaNamespace targetNamespace, String targetType, String topLevelQueryField, String targetFieldMatchingQueryArgument) {
            this.targetNamespace = targetNamespace;
            this.targetType = targetType;
            this.topLevelQueryField = topLevelQueryField;
            this.targetFieldMatchingQueryArgument = targetFieldMatchingQueryArgument;
            return this;
        }

        public LinkBuilder removeInputField(boolean removeInputField) {
            this.removeInputField = removeInputField;
            return this;
        }

        /**
         * @return
         *
         * @deprecated use {@link #removeInputField(boolean)}
         */
        @Deprecated
        public LinkBuilder replaceFromField() {
            return removeInputField(true);
        }


        /**
         * @param queryArgumentName
         *
         * @return
         *
         * @deprecated use {@link #queryArgumentName}
         */
        @Deprecated
        public LinkBuilder argument(String queryArgumentName) {
            return queryArgumentName(queryArgumentName);
        }

        public LinkBuilder queryArgumentName(String queryArgumentName) {
            this.queryArgumentName = queryArgumentName;
            return this;
        }

        public LinkBuilder sourceNamespace(SchemaNamespace sourceNamespace) {
            this.sourceNamespace = sourceNamespace;
            return this;
        }

        public LinkBuilder targetNamespace(SchemaNamespace targetNamespace) {
            this.targetNamespace = targetNamespace;
            return this;
        }

        public LinkBuilder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        public LinkBuilder topLevelQueryField(String topLevelQueryField) {
            this.topLevelQueryField = topLevelQueryField;
            return this;
        }


        public LinkBuilder sourceInputFieldName(String sourceInputFieldName) {
            this.sourceInputFieldName = sourceInputFieldName;
            return this;
        }

        public LinkBuilder setNullable(boolean nullable) {
            this.nullable = nullable;
            return this;
        }

        public LinkBuilder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        public LinkBuilder newFieldName(String newFieldName) {
            this.newFieldName = newFieldName;
            return this;
        }

        public LinkBuilder targetFieldMatchingQueryArgument(String targetFieldMatchingQueryArgument) {
            this.targetFieldMatchingQueryArgument = targetFieldMatchingQueryArgument;
            return this;
        }

        public LinkBuilder noSchemaChangeNeeded(boolean noSchemeChangeNeeded) {
            this.noSchemeChangeNeeded = noSchemeChangeNeeded;
            return this;
        }

        public LinkBuilder customTransformation(CustomTransformation customTransformation) {
            this.customTransformation = customTransformation;
            return this;
        }

        public Link build() {
            requireNonNull(sourceNamespace);
            requireNonNull(sourceType);
            requireNonNull(targetNamespace);
            requireNonNull(targetType);
            requireNonNull(newFieldName);
            String topLevelQueryField = Optional.ofNullable(this.topLevelQueryField).orElse(newFieldName);
            String sourceInputFieldName = Optional.ofNullable(this.sourceInputFieldName).orElse(newFieldName);
            String queryArgumentName = Optional.ofNullable(this.queryArgumentName).orElse("id");
            String targetFieldMatchingQueryArgument = Optional.ofNullable(this.targetFieldMatchingQueryArgument).orElse(queryArgumentName);
            return new Link(
                    sourceNamespace,
                    sourceType,
                    targetNamespace,
                    targetType,
                    newFieldName,
                    topLevelQueryField,
                    queryArgumentName,
                    sourceInputFieldName,
                    targetFieldMatchingQueryArgument,
                    removeInputField,
                    nullable,
                    noSchemeChangeNeeded,
                    customTransformation);

        }

    }
}