package com.atlassian.bitbucket.zdu;

import com.atlassian.bitbucket.util.Version;

import javax.annotation.Nonnull;
import java.util.Optional;

/**
 * Describes a service for maintaining rolling upgrade state. A rolling upgrade can be performed by enabling upgrade
 * mode and then upgrading nodes to a higher patch version, one at a time. After all the nodes have been upgraded,
 * upgrade mode should be disabled to finalize the upgrade.
 *
 * @since 8.0
 */
public interface RollingUpgradeService {

    /**
     * Disables upgrade mode if all the cluster nodes are on same version. If this method is called when cluster nodes
     * have different versions, an exception is thrown. If upgrade mode is already disabled, calling this method has no
     * effect.
     *
     * @see #enableUpgradeMode()
     */
    void disableUpgradeMode();

    /**
     * Enables upgrade mode which allows cluster to be run in mixed mode i.e. nodes with higher patch version than
     * {@link #getOriginalVersion() original version} will be able to join cluster while some other nodes may still be
     * on {@link #getOriginalVersion()} original version. After this method is called, current version will be saved and
     * returned by subsequent calls to {@link #getOriginalVersion()}. If upgrade mode is already enabled, calling this
     * method has no effect.
     */
    void enableUpgradeMode();

    /**
     * Retrieve the application {@link Version version} when upgrade mode was enabled.
     * <p>
     * The version returned is the "base" version for the rolling upgrade.  During rolling upgrade, some or all
     * nodes might have been upgraded to newer version, but this value will represent the version before upgrade was
     * started on any node. If upgrade mode is not currently {@link #isUpgradeModeEnabled() enabled},
     * {@link Optional#empty()} will be returned.
     *
     * @return the application {@link Version version} when upgrade mode was enabled, or {@link Optional#empty()}
     *         if upgrade mode is not currently enabled
     */
    @Nonnull
    Optional<Version> getOriginalVersion();

    /**
     * @return if upgrade mode is currently enabled or not.
     * @see #enableUpgradeMode()
     */
    boolean isUpgradeModeEnabled();
}
