package com.atlassian.bitbucket.util;

import javax.annotation.Nonnull;

/**
 * Marker interface augmenting a callback which produces a {@link Page} of results during its processing.
 * <p>
 * Callbacks passed to SCM commands are primary candidates for implementing this interface. SCM implementations are then
 * able to potentially optimise the output the command produces based on the page of data being requested.
 * <p>
 * Note: Implementing this interface does not <i>mandate</i> different handling from components that feed data into the
 * callback; it <i>encourages</i> it, where possible. Paged callbacks still have to handle being provided results which
 * are not on the requested page, either before or after, gracefully.
 */
public interface PagedCallback {

    /**
     * Retrieves the {@link PageRequest} which defines the {@link Page} the callback is attempting to build, allowing
     * components which provide data to the callback to potentially assist in producing the desired page.
     *
     * @return the paged request defining the {@link Page} the callback is attempting to build
     */
    @Nonnull
    PageRequest getPageRequest();

    /**
     * May be called by whatever component feeds information to the callback to indicate that the bounds of the
     * {@link PageRequest} have been applied already. In other words, if this method is invoked, it means the
     * producer was able to optimise the output based on the paging this callback would have done and will feed
     * it only data that is on the desired page.
     * <p>
     * Note: To allow the callback to determine whether there is another page, one more row than the requested
     * page may be provided.
     */
    void paged();
}
