package com.atlassian.bitbucket.user;

import com.atlassian.bitbucket.ForbiddenException;
import com.atlassian.bitbucket.IntegrityException;
import com.atlassian.bitbucket.NoSuchEntityException;
import com.atlassian.bitbucket.event.user.UserErasedEvent;
import com.atlassian.bitbucket.license.LicenseLimitException;
import com.atlassian.bitbucket.mail.MailException;
import com.atlassian.bitbucket.mail.NoMailHostConfigurationException;
import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

/**
 * Provides methods for querying, <i>creating and updating</i> users and groups.
 * <p>
 * Many of the queries provided by this service are also provided by {@link UserService}. The queries provided here
 * are specifically geared toward <i>administration</i>, retrieving {@link DetailedGroup detailed groups} and
 * {@link DetailedUser detailed groups}. For non-administrative usages, like group and user pickers, the "normal"
 * queries from {@link UserService} should be used instead.
 *
 * @see UserService
 */
public interface UserAdminService {

    /**
     * Adds a user to one or more groups.
     *
     * @param username name of the user who will receive the new group memberships
     * @param groupNames names of the groups to add to the user
     * @throws ForbiddenException if one of the groups would grant {@link Permission#SYS_ADMIN SYS_ADMIN} permission
     *                            and the current user isn't a SYS_ADMIN; only SYS_ADMINs can grant SYS_ADMIN permission
     * @throws LicenseLimitException if one of the groups would grant {@link Permission#LICENSED_USER LICENSED_USER}
     *                               permission to more users than the active license allows
     * @throws NoSuchGroupException if any of the specified groups does not exist
     * @throws NoSuchUserException if the specified user does not exist
     */
    void addUserToGroups(@Nonnull String username, @Nonnull Set<String> groupNames)
            throws ForbiddenException, LicenseLimitException, NoSuchGroupException, NoSuchUserException;

    /**
     * Adds one or more users to a group.
     *
     * @param groupName names of the group to which the users will be added
     * @param usernames names of the users who will receive the new group membership
     * @throws ForbiddenException if the group would grant {@link Permission#SYS_ADMIN SYS_ADMIN} permission
     *                            and the current user isn't a SYS_ADMIN; only SYS_ADMINs can grant SYS_ADMIN permission
     * @throws LicenseLimitException if the additions would grant {@link Permission#LICENSED_USER LICENSED_USER}
     *                               permission to more users than the active license allows
     * @throws NoSuchGroupException if the specified group does not exist
     * @throws NoSuchUserException if any of the specified users does not exist
     */
    void addMembersToGroup(@Nonnull String groupName, @Nonnull Set<String> usernames)
            throws ForbiddenException, LicenseLimitException, NoSuchGroupException, NoSuchUserException;

    /**
     * Retrieves a flag indicating whether it is possible to create a group.
     *
     * @return {@code true} if any directory exists in which a group can be created; otherwise, {@code false}
     */
    boolean canCreateGroups();

    /**
     * Retrieves a flag indicating whether it is possible to update groups.
     *
     * @return {@code true} if any directory exists in which a group can be updated; otherwise, {@code false}
     */
    boolean canUpdateGroups();

    /**
     * Retrieves a flag indicating whether it is possible to create a user.
     *
     * @return {@code true} if any directory exists in which a user can be created; otherwise, {@code false}
     */
    boolean canCreateUsers();

    /**
     * Retrieves a flag indicating whether it is possible to delete a group.
     *
     * @return {@code true} if any directory exists in which a group can be deleted; otherwise {@code false}
     */
    boolean canDeleteGroups();

    /**
     * Retrieves a flag indicating whether the new user is able to set a password or not
     *
     * @return {@code true} if user is able to set a password; otherwise {@code false}
     */
    boolean newUserCanResetPassword();

    /**
     * Clears any CAPTCHA challenge that may constrain the user with the supplied username when they authenticate.
     * Additionally any counter or metric that contributed towards the user being issued the CAPTCHA challenge
     * (for instance too many consecutive failed logins) will also be reset.
     *
     * @param username the user whose CAPTCHA challenge should be cleared
     */
    void clearCaptchaChallenge(@Nonnull String username);

    /**
     * Creates a new group.
     *
     * @param groupName the name for the new group
     * @return the new group
     * @throws IntegrityException if a group with the same already exists
     */
    @Nonnull
    DetailedGroup createGroup(@Nonnull String groupName) throws IntegrityException;

    /**
     * Creates a new {@link UserType#SERVICE service} user. Service users cannot be added to any groups. The user
     * is created without any permissions.
     * <p>
     * It is suggested that consumers use the {@link ApplicationUser#getId() ID} of the user created as the foreign key for
     * related content because their name, slug and other properties can be updated.
     *
     * @param userRequest the user's details
     * @return the newly created service user
     * @throws IntegrityException if a service user with the provided username already exists
     */
    @Nonnull
    ServiceUser createServiceUser(@Nonnull ServiceUserCreateRequest userRequest) throws IntegrityException;

    /**
     * Creates a new user and adds them to the default group, if it exists. If the user should be created without being
     * added to the default group, use {@link #createUser(String, String, String, String, boolean)} instead.
     * <p>
     * A non-{@code null}, non-blank {@code password} must be provided. It may be further vetted by the user store,
     * for example by applying complexity restrictions. Alternatively, if an e-mail server has been configured, new
     * users can be created {@link #createUserWithGeneratedPassword(String, String, String) with generated passwords},
     * allowing new users to set their own password when they first access the system.
     *
     * @param username     the {@link ApplicationUser#getName() username} for the new user
     * @param password     the user's initial password, which may not be {@code null} or empty
     * @param displayName  the {@link ApplicationUser#getDisplayName() display name} for the new user
     * @param emailAddress the {@link ApplicationUser#getEmailAddress() e-mail address} for the new user
     * @throws IntegrityException    if a user with the same username already exists
     * @throws LicenseLimitException if the new user can not be added to the default group because it would grant
     *                               {@link Permission#LICENSED_USER LICENSED_USER} permission to more users than
     *                               the active license allows
     * @see #createUser(String, String, String, String, boolean)
     */
    void createUser(@Nonnull String username, @Nonnull String password, @Nonnull String displayName,
                    @Nonnull String emailAddress)
            throws IntegrityException, LicenseLimitException;

    /**
     * Creates a new user and optionally adds them to the default group, if it exists.
     * <p>
     * A non-{@code null}, non-blank {@code password} must be provided. It may be further vetted by the user store,
     * for example by applying complexity restrictions. Alternatively, if an e-mail server has been configured, new
     * users can be created {@link #createUserWithGeneratedPassword(String, String, String) with generated passwords},
     * allowing new users to set their own password when they first access the system.
     * <p>
     * This method is not intended to be exposed via services like REST for general consumption. It exists to satisfy
     * specific use cases where the system, or a plugin, may wish to create a user <i>without</i> adding them to the
     * default group. Generally it is expected that new users should be added to the default group, as it defines the
     * set of "common" permissions for all users of the system.
     *
     * @param username     the {@link ApplicationUser#getName() username} for the new user
     * @param password     the user's initial password, which may not be {@code null} or empty
     * @param displayName  the {@link ApplicationUser#getDisplayName() display name} for the new user
     * @param emailAddress the {@link ApplicationUser#getEmailAddress() e-mail address} for the new user
     * @param addToDefaultGroup true if the new user should be added to the default group
     * @throws IntegrityException    if a user with the same username already exists
     * @throws LicenseLimitException if the new user can not be added to the default group because it would grant
     *                               {@link Permission#LICENSED_USER LICENSED_USER} permission to more users than
     *                               the active license allows
     * @see #createUser(String, String, String, String)
     */
    void createUser(@Nonnull String username, @Nonnull String password, @Nonnull String displayName,
                    @Nonnull String emailAddress, boolean addToDefaultGroup)
            throws IntegrityException, LicenseLimitException;

    /**
     * Creates a new user with a randomly-generated password. An e-mail notification will be sent to the new user's
     * e-mail address with instructions on how to reset their password and finish activating their account.
     *
     * @param username     the {@link ApplicationUser#getName() username} for the new user
     * @param displayName  the {@link ApplicationUser#getDisplayName() display name} for the new user
     * @param emailAddress the {@link ApplicationUser#getEmailAddress() e-mail address} for the new user
     * @throws IntegrityException    if a user with the same username already exists or if the user will be created
     *                               in a directory that does not allow passwords to be reset
     * @throws LicenseLimitException if the new user can not be added to the default group because it would grant
     *                               {@link Permission#LICENSED_USER LICENSED_USER} permission to more users than
     *                               the active license allows
     * @throws MailException         if the e-mail notification could not be sent to the created user to allow them
     *                               to set their initial password
     * @throws NoMailHostConfigurationException if no e-mail server has been configured
     */
    void createUserWithGeneratedPassword(@Nonnull String username, @Nonnull String displayName,
                                         @Nonnull String emailAddress)
            throws IntegrityException, LicenseLimitException, MailException;

    /**
     * Deletes a group. Deleting a group will also revoke all permissions which had been granted to that group.
     * <p>
     * Note: Future releases may remove the return value from this method. For best portability across versions, the
     * return value for this method should not be used.
     *
     * @param groupName the name of the group to delete
     * @return details for the deleted group
     * @throws ForbiddenException   if the group grants {@link Permission#SYS_ADMIN SYS_ADMIN} permission but the
     *                              current user is not a SYS_ADMIN
     * @throws IntegrityException   if deleting the group would revoke the {@link Permission#ADMIN ADMIN} or
     *                              {@link Permission#SYS_ADMIN SYS_ADMIN} permissions of the current user
     * @throws NoSuchGroupException if the group does not exist
     */
    @Nonnull
    DetailedGroup deleteGroup(@Nonnull String groupName)
            throws ForbiddenException, IntegrityException, NoSuchGroupException;

    /**
     * Deletes a user. Deleting a user will also revoke all permissions which had been granted to that user, as well
     * as invalidating any cached authentication tokens they may have.
     * <p>
     * Note: Future releases may remove the return value from this method. For best portability across versions, the
     * return value for this method should not be used.
     *
     * @param username the {@link ApplicationUser#getName() username} of the user to delete
     * @return details for the deleted user
     * @throws ForbiddenException  if the user to delete is a {@link Permission#SYS_ADMIN SYS_ADMIN} and the current
     *                             user is not
     * @throws IntegrityException  if the user to delete is the current user
     * @throws NoSuchUserException if the user does not exist
     */
    @Nonnull
    DetailedUser deleteUser(@Nonnull String username)
            throws ForbiddenException, IntegrityException, NoSuchUserException;

    /**
     * Erases personally identifying user data for a deleted user. References in the application to the original
     * username will be either removed or updated to a new non-identifying username.
     * <p>
     * A {@link UserErasedEvent} is then raised once erasure has completed.
     * <p>
     * Erasing user data is <strong>irreversible</strong> and may lead to a degraded user experience. This method should
     * not be used as part of a standard user deletion and cleanup process.
     *
     * @param username the {@link ApplicationUser#getName() username} of the user to erase data for
     * @return the new identifier for the erased user
     * @throws IllegalUserStateException if the supplied {@code username} is the username of a non-deleted user
     * @throws NoSuchUserException if the supplied {@code username} does not exist
     * @throws UserErasureException if some of the delegated user erasure operations failed
     * @see UserErasedEvent
     * @since 5.16
     */
    @Nonnull
    String eraseUser(@Nonnull String username)
            throws IllegalUserStateException, NoSuchUserException, UserErasureException;

    /**
     * Retrieves a page of groups with full {@link DetailedGroup details}.
     * <p>
     * Note: To filter the retrieved groups by name, use {@link #findGroupsByName(String, PageRequest)} instead.
     *
     * @param pageRequest defines the page of groups to retrieve
     * @return the requested page of groups, which may be empty but never {@code null}
     * @see #findGroupsByName(String, PageRequest)
     * @see UserService#findGroups(PageRequest)
     */
    @Nonnull
    Page<DetailedGroup> findGroups(@Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of groups with full {@link DetailedGroup details}, optionally filtering the returned results
     * to those containing the specified {@code groupName}. If the provided {@code groupName} is {@code null}, this
     * method behaves identically to {@link #findGroups(PageRequest)}.
     *
     * @param groupName   0 or more characters to apply as a filter on returned groups
     * @param pageRequest defines the page of groups to retrieve
     * @return the requested page of groups, potentially filtered, which may be empty but never {@code null}
     * @see #findGroups(PageRequest)
     * @see UserService#findGroupsByName(String, PageRequest)
     */
    @Nonnull
    Page<DetailedGroup> findGroupsByName(@Nullable String groupName, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of groups which the specified user is a member of, with full {@link DetailedGroup details},
     * optionally filtering the returned results to those containing the specified {@code groupName}. If the provided
     * {@code groupName} is {@code null} or empty groups will not be filtered. If no user exists with the specified
     * {@code username}, no groups will be returned.
     *
     * @param username    the <i>exact</i> name of the user to retrieve groups for
     * @param groupName   0 or more characters to apply as a filter on returned groups
     * @param pageRequest defines the page of groups to retrieve
     * @return the requested page of groups, which may be empty but never {@code null}
     * @see #findGroupsWithoutUser(String, String, PageRequest)
     * @see #findUsersWithGroup(String, String, PageRequest)
     * @see UserService#findGroupsByUser(String, PageRequest)
     */
    @Nonnull
    Page<DetailedGroup> findGroupsWithUser(@Nonnull String username, @Nullable String groupName,
                                           @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of groups which the specified user is <i>not</i> a member of, with full {@link DetailedGroup
     * details}, optionally filtering the returned results to those containing the specified {@code groupName}. If the
     * provided {@code groupName} is {@code null} or empty groups will not be filtered. If no user exists with the
     * specified {@code username}, no groups will be returned.
     *
     * @param username    the <i>exact</i> name of the user to retrieve groups for
     * @param groupName   0 or more characters to apply as a filter on returned groups
     * @param pageRequest defines the page of groups to retrieve
     * @return the requested page of groups, which may be empty but never {@code null}
     * @see #findGroupsWithUser(String, String, PageRequest)
     * @see #findUsersWithGroup(String, String, PageRequest)
     * @see UserService#findGroupsByUser(String, PageRequest)
     */
    @Nonnull
    Page<DetailedGroup> findGroupsWithoutUser(@Nonnull String username, @Nullable String groupName,
                                              @Nonnull PageRequest pageRequest);

    /**
     * Find a password reset request using the token generated by {@link #requestPasswordReset(String)}.
     *
     * @param token token identifying the password reset request
     * @return the user matching the password reset request or {@code null} if no request matches the token or if
     *         the request has expired
     * @see #requestPasswordReset(String)
     */
    @Nullable
    DetailedUser findUserByPasswordResetToken(@Nonnull String token);

    /**
     * Retrieves a page of users with full {@link DetailedUser details}.
     * <p>
     * Note: To filter the retrieved users by name, use {@link #findUsersByName(String, PageRequest)} instead.
     *
     * @param pageRequest defines the page of users to retrieve
     * @return the requested page of users, which may be empty but never {@code null}
     * @see #findUsersByName(String, PageRequest)
     * @see UserService#findUsers(PageRequest)
     */
    @Nonnull
    Page<DetailedUser> findUsers(@Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of users, optionally filtering the returned results to those containing the specified
     * {@code username}. If the provided {@code username} is {@code null}, this method behaves identically to
     * {@link #findUsers(PageRequest)}. Otherwise, the {@code username} is matched against:
     * <ul>
     *     <li>{@link ApplicationUser#getDisplayName() Display names}</li>
     *     <li>{@link ApplicationUser#getEmailAddress() E-mail addresses}</li>
     *     <li>{@link ApplicationUser#getName() Usernames}</li>
     * </ul>
     *
     * @param username    0 or more characters to apply as a filter on returned users
     * @param pageRequest defines the page of users to retrieve
     * @return the requested page of users, potentially filtered, which may be empty but never {@code null}
     * @see #findUsers(PageRequest)
     * @see UserService#findUsersByName(String, PageRequest)
     */
    @Nonnull
    Page<DetailedUser> findUsersByName(@Nullable String username, @Nonnull PageRequest pageRequest);

    /**
     * Find the users within a group that match the page request.
     *
     * @param groupName   name of the group the users must belong to
     * @param username    0 or more characters to apply as a filter on returned users
     * @param pageRequest page request
     * @return page of users matching search criteria
     * @return a page of {@link ApplicationUser}
     */
    @Nonnull
    Page<DetailedUser> findUsersWithGroup(@Nonnull String groupName, @Nullable String username,
                                          @Nonnull PageRequest pageRequest);

    /**
     * Find the users outside a group that match the page request.
     *
     * @param groupName   name of the group the users must not belong to
     * @param username    0 or more characters to apply as a filter on returned users
     * @param pageRequest page request
     * @return a page of {@link ApplicationUser}
     */
    @Nonnull
    Page<DetailedUser> findUsersWithoutGroup(@Nonnull String groupName, @Nullable String username,
                                             @Nonnull PageRequest pageRequest);

    /**
     * Retrieves full {@link DetailedUser details} for the user with the specified {@link ApplicationUser#getName() username},
     * or {@code null} if no such user exists.
     *
     * @param username the <i>exact</i> username of the user to retrieve
     * @return full {@link DetailedUser details} for the specified user, or {@code null} if no user
     */
    @Nullable
    DetailedUser getUserDetails(@Nonnull String username);

    /**
     * Retrieves full {@link DetailedUser details} for the specified {@link ApplicationUser user}. This method is intended to
     * "promote" from a {@link ApplicationUser} to a {@link DetailedUser}, providing access to mutability details and other
     * information for the user.
     *
     * @param user the user to retrieve details for
     * @return full {@link DetailedUser details} for the specified user
     * @throws NoSuchUserException if the specified {@code user} does not exist in the underlying user store
     */
    @Nonnull
    DetailedUser getUserDetails(@Nonnull ApplicationUser user);

    /**
     * Removes a user from a group.
     *
     * @param groupName name of the group the user will be removed from
     * @param username  name of the user to remove from the group
     * @throws ForbiddenException   if the group grants {@link Permission#SYS_ADMIN SYS_ADMIN} permission but the
     *                              current user is not a SYS_ADMIN
     * @throws IntegrityException   if the current user belongs to the specified group and removing them from the
     *                              group would revoke their {@link Permission#SYS_ADMIN SYS_ADMIN} or
     *                              {@link Permission#ADMIN ADMIN} permission
     * @throws NoSuchGroupException if the specified group does not exist
     * @throws NoSuchUserException if the specified user does not exist
     */
    void removeUserFromGroup(@Nonnull String groupName, @Nonnull String username)
            throws ForbiddenException, IntegrityException, NoSuchGroupException, NoSuchUserException;

    /**
     * Change the name of a user
     *
     * @param currentUsername the current name of the user
     * @param newUsername     the new name of the user
     * @return the newly renamed user
     * @throws NoSuchUserException if the specified user does not exist
     * @throws IntegrityException  if a user already exists in the directory with the new name
     * @throws ForbiddenException  if renaming is not supported by the directory the target user belongs to
     */
    @Nonnull
    DetailedUser renameUser(@Nonnull String currentUsername, @Nonnull String newUsername);

    /**
     * Generates a unique token which can be used to perform a {@link #resetPassword(String, String) password reset}
     * for the specified user and e-mails it to the address associated with their account.
     *
     * @param username username of the user
     * @throws MailException if the e-mail notification could not be sent to the user (ex: the mail server is down)
     * @throws NoMailHostConfigurationException if no e-mail server has been configured
     * @throws NoSuchUserException if the user does not exist
     */
    void requestPasswordReset(@Nonnull String username) throws MailException, NoSuchUserException;

    /**
     * Resets the password for the {@link ApplicationUser user} associated with the specified token to the provided value.
     *
     * @param token    the token identifying the user whose password should be reset
     * @param password the new password for the user
     * @throws InvalidPasswordResetTokenException if no user matches the specified token
     */
    void resetPassword(@Nonnull String token, @Nonnull String password)
            throws InvalidPasswordResetTokenException, NoSuchEntityException;

    /**
     * Updates the password of the specified user.
     * <p>
     * Note: A {@link Permission#ADMIN ADMIN} cannot update the password of a {@link Permission#SYS_ADMIN SYS_ADMIN}.
     *
     * @param username    the user's username
     * @param newPassword the user's new password
     */
    void updatePassword(@Nonnull String username, @Nonnull String newPassword);

    /**
     * Updates the details of the specified {@link ServiceUser user}
     *
     * @param request  the user's new details
     * @return the updated user
     */
    @Nonnull
    ServiceUser updateServiceUser(@Nonnull ServiceUserUpdateRequest request);

    /**
     * Updates the {@link ApplicationUser#getDisplayName() display name} and {@link ApplicationUser#getEmailAddress()
     * e-mail address} of the specified user.
     * <p>
     * Note: A {@link Permission#ADMIN ADMIN} cannot update the details of a {@link Permission#SYS_ADMIN SYS_ADMIN}.
     *
     * @param username     the user's username
     * @param displayName  the user's new display name
     * @param emailAddress the user's new email address
     * @return the updated user
     */
    @Nonnull
    DetailedUser updateUser(@Nonnull String username, @Nonnull String displayName, @Nonnull String emailAddress);

    /**
     * Validate if a username can be erased.
     * <p>
     * A username is only valid for erasure if it exists as the username of a deleted user. This method will throw
     * an appropriate exception if the user for the supplied username is invalid for erasure.
     *
     * @param username the {@link ApplicationUser#getName() username} of the user to validate erasability for
     * @throws IllegalUserStateException if the supplied {@code username} is the username of a non-deleted user
     * @throws NoSuchUserException       if the supplied {@code username} does not exist
     *
     * @since 5.16
     */
    void validateErasable(@Nonnull String username) throws IllegalUserStateException, NoSuchUserException;
}
