package com.atlassian.bitbucket.scm.bulk;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * The context for a bulk content callback that is passed at the start.
 *
 * @since 4.2
 */
public class BulkContentContext {

    private final String sinceCommitId;
    private final String untilCommitId;

    private BulkContentContext(Builder builder) {
        sinceCommitId = builder.sinceCommitId;
        untilCommitId = builder.untilCommitId;
    }

    /**
     * @return optional ID of the "since" commit; if specified only contents that changed since that commit ID are streamed, otherwise all contents are streamed
     */
    @Nonnull
    public Optional<String> getSinceCommitId() {
        return ofNullable(sinceCommitId);
    }

    /**
     * @return ID of the commit whose file contents are be streamed
     */
    @Nonnull
    public String getUntilCommitId() {
        return untilCommitId;
    }

    public static class Builder {

        private final String untilCommitId;

        private String sinceCommitId;

        public Builder(@Nonnull String untilCommitId) {
            this.untilCommitId = requireNonNull(untilCommitId, "untilCommitId");
        }

        @Nonnull
        public Builder sinceCommitId(@Nullable String value) {
            sinceCommitId = value;

            return this;
        }

        @Nonnull
        public BulkContentContext build() {
            return new BulkContentContext(this);
        }
    }
}
