package com.atlassian.bitbucket.scm.bulk;

import javax.annotation.Nonnull;
import java.io.InputStream;

/**
 * Callback for streamed file content.
 * <p>
 * Note that there is no guarantee about the order in which the files are handed to the callback.
 *
 * @since 4.2
 */
@FunctionalInterface
public interface BulkContentCallback {

    /**
     * For each added, modified or deleted file, this method is called once with the metadata. This allows the callback
     * to request content if it's interested in it or stop processing. If the callback signals that it wants the file's
     * content, it will be provided to {@link #onFile}.
     *
     * @param file information about the file
     * @return an instruction to the caller about how to proceed
     */
    @Nonnull
    default BulkContentDisposition accept(@Nonnull BulkFile file) {
        return BulkContentDisposition.STREAM;
    }

    /**
     * Called at the end.
     *
     * @param summary provides details about the command execution
     */
    default void onEnd(@Nonnull BulkContentSummary summary) {
    }

    /**
     * For each file where the content was requested with {@link #accept}, this method is called.
     *
     * @param file    information about the file
     * @param content the input stream of the contents; must be consumed before method returns (it won't stay usable after)
     */
    void onFile(@Nonnull BulkFile file, @Nonnull InputStream content);

    /**
     * Called before any other methods are called.
     *
     * @param context provides details about the request
     */
    default void onStart(@Nonnull BulkContentContext context) {
    }
}
