package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.commit.CommitService;
import com.atlassian.bitbucket.commit.LastModifiedCallback;
import com.atlassian.bitbucket.commit.LastModifiedRequest;
import com.atlassian.bitbucket.content.ArchiveRequest;
import com.atlassian.bitbucket.content.ContentService;
import com.atlassian.bitbucket.content.EditFileRequest;
import com.atlassian.bitbucket.content.PatchRequest;
import com.atlassian.bitbucket.io.TypeAwareOutputSupplier;
import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.repository.RefService;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.Tag;
import com.atlassian.bitbucket.scm.signed.SignedObjectCallback;
import com.atlassian.bitbucket.scm.signed.SignedObjectsParameters;

import javax.annotation.Nonnull;

/**
 * Creates {@link Command commands} which provide optional SCM functionality such as merging two branches or updating
 * the default branch. Each method accepts a set of {@code CommandParameters} which are used to control the command's
 * behaviour and output.
 * <p>
 * Each command on this factory has an associated {@link ScmFeature SCM feature}. Prior to using any of these commands,
 * callers should {@link ScmService#isSupported(Repository, ScmFeature) check if the feature is supported}. Failure to
 * do so will result in {@link FeatureUnsupportedScmException}s when requesting unsupported commands.
 *
 * @since 4.6
 */
public interface ScmExtendedCommandFactory {

    /**
     * Streams an archive of the specified {@link ArchiveCommandParameters#getCommitId() commit} in the requested
     * {@link ArchiveCommandParameters#getFormat() format}.
     * <p>
     * The provided commit ID may be:
     * <ul>
     *     <li>A {@link Branch#getId() branch ID} or {@link Branch#getDisplayId() display ID}</li>
     *     <li>A {@link Commit#getId() commit ID}
     *         <ul>
     *             <li>{@link Commit#getDisplayId() Display IDs} are <i>not supported</i>; there is no guarantee
     *             a given display ID will be unique within the repository</li>
     *         </ul>
     *     </li>
     *     <li>A {@link Tag#getId() tag ID} or {@link Tag#getDisplayId() display ID}</li>
     * </ul>
     * <p>
     * This operation is optional and may not be supported by all SCM implementations. Calling this method on an
     * SCM which does not support it will result in a {@link FeatureUnsupportedScmException}.
     *
     * @param parameters     parameters describing the commit to archive and the format to produce the archive in
     * @param outputSupplier a supplier which, when given the archive content type, will provide an output stream
     * @return a command which, when executed, will stream an archive of the repository at the specified commit
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support archiving repositories
     * @see ContentService#streamArchive(ArchiveRequest, TypeAwareOutputSupplier)
     * @see ScmFeature#ARCHIVE
     * @since 5.1
     */
    @Nonnull
    Command<Void> archive(@Nonnull ArchiveCommandParameters parameters,
                          @Nonnull TypeAwareOutputSupplier outputSupplier);

    /**
     * Adds a new file or updates the content of an existing file as described in the {@code parameters}.
     * <p>
     * This operation is optional and may not be supported by all SCM implementations. Calling this method on an
     * SCM which does not support it will result in a {@link FeatureUnsupportedScmException}.
     *
     * @param parameters parameters describing the file to be edited
     * @return a command which, when executed, will edit the content of a file
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support editing a file.
     * @see ContentService#editFile(EditFileRequest)
     * @see ScmFeature#EDIT_FILE
     * @since 4.13
     */
    @Nonnull
    Command<Commit> editFile(@Nonnull EditFileCommandParameters parameters);

    /**
     * Streams the latest commit to modify each file in the specified {@link LastModifiedCommandParameters#getPath()
     * path}, using the provided {@link LastModifiedCommandParameters#getCommitId() commit} as the starting point for the
     * traversal.
     * <p>
     * This operation is optional and may not be supported by all SCM implementations. Calling this method on an
     * SCM which does not support it will result in a {@link FeatureUnsupportedScmException}.
     *
     * @param parameters parameters describing the path to stream modifications for, and the starting commit for
     *                   the traversal
     * @param callback   a callback to receive the latest commit for files in the specified path
     * @return a command which, when executed, will stream the latest commit to modify each file in the specified path
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support streaming modifications
     * @see CommitService#streamLastModified(LastModifiedRequest, LastModifiedCallback)
     * @see ScmFeature#LAST_MODIFIED
     */
    @Nonnull
    Command<Void> lastModified(@Nonnull LastModifiedCommandParameters parameters,
                               @Nonnull LastModifiedCallback callback);

    /**
     * Merges the specified {@code fromCommitId} into the specified {@code toBranch}.
     * <p>
     * The provided {@code fromCommitId} may be:
     * <ul>
     *     <li>A 40-byte hash (Note: Short hashes are <i>not supported</i>)</li>
     *     <li>A fully-qualified branch name</li>
     *     <li>A short branch name</li>
     * </ul>
     * The provided {@code toBranch}, as the parameter name suggests, <i>must</i> be a branch name, either fully
     * qualified or short; using a hash (full or short) is not supported.
     * <p>
     * If the {@link ScmService#isSupported(Repository, ScmFeature) SCM supports} {@link ScmFeature#CROSS_REPOSITORY
     * cross-repository} merges, a different {@link MergeCommandParameters#getFromRepository() from repository} may
     * be provided. Otherwise, it must be omitted or match the target repository.
     * <p>
     * This operation is optional and may not be supported by all SCM implementations. Calling this method on an
     * SCM which does not support it will result in a {@link FeatureUnsupportedScmException}.
     * <p>
     * <b>Note</b>: This method is not exposed via the service API. It is only available directly on the SCM API.
     *
     * @param parameters parameters describing the branches to merge
     * @return a command which, when executed, will merge the specified commit into the target branch
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support merging branches, or if a
     *                                        cross-repository merge was requested and the SCM doesn't support
     *                                        {@link ScmFeature#CROSS_REPOSITORY cross-repository} merges
     * @see ScmFeature#MERGE
     */
    @Nonnull
    Command<Branch> merge(@Nonnull MergeCommandParameters parameters);

    /**
     * Streams a patch for a given {@link PatchCommandParameters#getUntilId() commit} or commit range.
     * <p>
     * If {@link PatchCommandParameters#isAllAncestors()} is true then the patch includes all ancestors of the
     * {@link PatchCommandParameters#getUntilId() "until"} revision. Otherwise the patch starts from the
     * {@link PatchCommandParameters#getSinceId() "since"} revision, if specified, or from the parent of the
     * {@link PatchCommandParameters#getUntilId() "until"} revision.
     * <p>
     * This operation is optional and may not be supported by all SCM implementations. Calling this method on an SCM
     * which does not support it will result in a {@link FeatureUnsupportedScmException}.
     *
     * @param parameters     parameters describing the commit for which to generate the patch
     * @param outputSupplier a supplier which will provide an output stream
     * @return a command which, when executed, will stream a patch of the diff at the specified commit
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support patches for repositories
     * @see ContentService#streamPatch(PatchRequest, TypeAwareOutputSupplier)
     * @see ScmFeature#PATCH
     * @since 6.7
     */
    @Nonnull
    Command<Void> patch(@Nonnull PatchCommandParameters parameters, @Nonnull TypeAwareOutputSupplier outputSupplier);

    /**
     * Pushes the repository to the {@link PushCommandParameters#getRemoteUrl() URL specified in the parameters}.
     *
     * @param parameters parameters describing the target of the push, optional authentication and which refs should
     *                   be included
     * @param callback   a callback that will receive ref updates, ref failures and progress updates
     * @return a command which, when executed, will perform a push of the repository to the URL specified
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support pushing
     * @see ScmFeature#PUSH
     * @since 7.11
     */
    @Nonnull
    Command<Void> push(@Nonnull PushCommandParameters parameters, @Nonnull PushCallback callback);

    /**
     * Retrieves requested objects and if the object is signed, provides the signature and signed content to the
     * callback.
     *
     * @param parameters describes which objects to retrieve
     * @param callback callback to receive the requested objects
     * @return a command which, when executed, will retrieve the requested objects and provides them to the provided
     *         callback
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support retrieving signed objects
     * @see ScmFeature#SIGNED_OBJECTS
     *
     * @since 5.1
     */
    @Nonnull
    Command<Void> signedObjects(@Nonnull SignedObjectsParameters parameters, @Nonnull SignedObjectCallback callback);

    /**
     * Sets the default branch for the repository to the specified branch.
     * <p>
     * This operation is optional and may not be supported by all SCM implementations. Calling this method on an
     * SCM which does not support it will result in a {@link FeatureUnsupportedScmException}.
     *
     * @param parameters parameters describing the branch to make the default
     * @return a command which, when executed, will update the repository's default branch
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support setting the default branch
     * @see RefService#setDefaultBranch(Repository, String)
     * @see ScmFeature#UPDATE_DEFAULT_BRANCH
     */
    @Nonnull
    Command<Void> updateDefaultBranch(@Nonnull UpdateDefaultBranchCommandParameters parameters);
}
