package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.PatchRequest;
import com.atlassian.bitbucket.repository.Repository;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Describes a commit for which to generate a {@link ScmExtendedCommandFactory#patch patch}.
 * <p>
 * In addition to specifying the commit for which to generate a patch, callers can provide a
 * {@link #getSinceId() since ID} revision as the base of the patch, or set {@link #isAllAncestors() all ancestors} to
 * true.
 *
 * @since 6.7
 */
public class PatchCommandParameters extends AbstractCommandParameters {

    private final boolean allAncestors;
    private final Repository secondaryRepository;
    private final String sinceId;
    private final String untilId;

    private PatchCommandParameters(Builder builder) {
        allAncestors = builder.allAncestors;
        secondaryRepository = builder.secondaryRepository;
        sinceId = builder.sinceId;
        untilId = builder.untilId;
    }

    @Nullable
    public Repository getSecondaryRepository() {
        return secondaryRepository;
    }

    /**
     * Retrieves the base revision from which to generate the patch. If omitted the parent revision of the
     * {@link #getUntilId() until ID} revision is used.
     *
     * @return the base revision from which to generate the patch
     */
    @Nullable
    public String getSinceId() {
        return sinceId;
    }

    @Nonnull
    public String getUntilId() {
        return untilId;
    }

    /**
     * @return whether or not to generate a patch which includes all the ancestors of {@link #getUntilId() until ID}.
     *         If true, the value provided by {@link #getSinceId() since ID} is ignored.
     */
    public boolean isAllAncestors() {
        return allAncestors;
    }

    public static class Builder {

        private boolean allAncestors;
        private Repository secondaryRepository;
        private String sinceId;
        private String untilId;

        public Builder() {
        }

        public Builder(@Nonnull PatchRequest patchRequest) {
            requireNonNull(patchRequest, "patchRequest");
            allAncestors = patchRequest.isAllAncestors();
            secondaryRepository = patchRequest.getSecondaryRepository();
            sinceId = patchRequest.getSinceId();
            untilId = patchRequest.getUntilId();
        }

        @Nonnull
        public PatchCommandParameters.Builder allAncestors(boolean value) {
            allAncestors = value;

            return this;
        }

        @Nonnull
        public PatchCommandParameters build() {
            if (StringUtils.isBlank(untilId)) {
                throw new IllegalStateException("An \"untilId\" is required to build a patch");
            }

            return new PatchCommandParameters(this);
        }

        @Nonnull
        public PatchCommandParameters.Builder secondaryRepository(@Nullable Repository value) {
            secondaryRepository = value;

            return this;
        }

        @Nonnull
        public PatchCommandParameters.Builder sinceId(@Nullable String value) {
            sinceId = value;

            return this;
        }

        @Nonnull
        public PatchCommandParameters.Builder untilId(@Nonnull String value) {
            //Nullability is enforced in build(); "value" is annotated @Nonnull to help callers avoid errors
            untilId = value;

            return this;
        }
    }
}
