package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.ChangeContext;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

import static java.util.Objects.requireNonNull;

@SuppressWarnings("unused") //Unused members are still public API
public class AbstractChangesCommandParameters extends AbstractCommandParameters {

    private final int maxChanges;
    private final Set<String> paths;

    /**
     * @param builder a builder containing the assembled parameters
     * @since 5.4
     */
    protected AbstractChangesCommandParameters(AbstractBuilder<?> builder) {
        maxChanges = builder.maxChanges;
        paths = builder.paths.build();
    }

    public int getMaxChanges() {
        return maxChanges;
    }

    @Nonnull
    public Set<String> getPaths() {
        return paths;
    }

    public boolean hasPaths() {
        return !paths.isEmpty();
    }

    /**
     * Appends common attributes to the provided {@link ChangeContext.Builder} before
     * building it and returning the final {@link ChangeContext}.
     *
     * @param builder the starting builder to apply common attributes to before building
     * @return the {@link ChangeContext} from the provided builder
     */
    @Nonnull
    protected ChangeContext toContext(@Nonnull ChangeContext.Builder builder) {
        return requireNonNull(builder, "builder")
                .maxChanges(getMaxChanges())
                .build();
    }

    public abstract static class AbstractBuilder<B extends AbstractBuilder<B>> {

        private final ImmutableSet.Builder<String> paths;

        private int maxChanges;

        protected AbstractBuilder() {
            paths = ImmutableSet.builder();
        }

        @Nonnull
        public B maxChanges(int value) {
            maxChanges = value;

            return self();
        }

        @Nonnull
        public B path(@Nullable String value) {
            addIf(NOT_BLANK, paths, value);

            return self();
        }

        @Nonnull
        public B paths(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, paths, values);

            return self();
        }

        @Nonnull
        public B paths(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, paths, value, values);

            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
