package com.atlassian.bitbucket.request;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.Callable;

/**
 * Provides information about a request, such as the requestId, sessionId, remoteAddress and more. {@code RequestContext}
 * is currently used as an abstraction for HTTP and SSH requests, but can easily be used for other protocols as well.
 *
 * @see RequestManager
 */
public interface RequestContext extends RequestInfoProvider {

    /**
     * Registers a cleanup task that is executed when the current request ends. Note that the callback will <em>only</em>
     * be called for the current request.
     *
     * @param callback callback that should be called when the request ends
     * @throws IllegalStateException if no request is {@link #isActive() active}
     */
    void addCleanupCallback(@Nonnull Runnable callback);

    /**
     * Adds a label to the request context that will be included in the access logs. Use this purely for diagnostic /
     * analysis purposes.
     *
     * @param label the label to add
     * @throws IllegalStateException if no request is {@link #isActive() active}
     */
    void addLabel(@Nonnull String label);

    /**
     * Returns a highly unique (though not guaranteeably unique) request ID.
     * <p>
     * The request ID contains:
     * <ol>
     *     <li>The minute of the current day</li>
     *     <li>The number of requests, including the current one, which have been serviced by the application since it
     *     was started</li>
     *     <li>The number of requests which were being processed concurrently at the time the ID was generated</li>
     * </ol>
     * These fields are separated by an "x". The hour of the day portion resets each night, and the concurrency count
     * rises and falls with the load on the server. The request count is monotonically increasing until the {@code long}
     * for the counter wraps (which at 1,000 requests per second will take 252 million years).
     * <p>
     * It is worth noting that the uniqueness period required for request IDs is 1 day, which is the period at which
     * log files are rotated. The goal of this ID is not to be universally unique so much as it is to allow support to
     * easily trace the logging related to a single request, within the log files.
     *
     * @return the generated request ID
     * @throws IllegalStateException if no request is {@link #isActive() active}
     */
    @Nonnull
    String getId();

    /**
     * @return {@code true} if the request is in progress. {@code false} if there is no current request or the request
     *         has finished
     */
    boolean isActive();

    /**
     * @since 4.4
     * @return the number of bytes read from the client for this request
     */
    long getBytesRead();

    /**
     * @since 4.4
     * @return the number of bytes written to the client for this request
     */
    long getBytesWritten();

    /**
     * @return a {@link Duration} representing the amount of wall time taken to process this request or
     * {@link Optional#empty()} if the duration has not been set
     * @since 4.4
     */
    @Nonnull
    Optional<Duration> getDuration();

    /**
     * Report on the number of bytes read for this request.
     * @param bytesRead the number of bytes read
     * @throws IllegalStateException if no request is {@link #isActive() active}
     */
    void setBytesRead(long bytesRead);

    /**
     * Report on the number of bytes written for this request.
     * @param bytesWritten the number of bytes written
     * @throws IllegalStateException if no request is {@link #isActive() active}
     */
    void setBytesWritten(long bytesWritten);

    /**
     * The response code of the request.
     * HTTP(S) requests the HTTP response code is used.
     * SSH requests the exit code of the git process is used.
     * @param responseCode the request response code
     * @throws IllegalStateException if no request is {@link #isActive() active}
     */
    void setResponseCode(int responseCode);

    /**
     * Wraps the provided {@link Callable} so that it executes with this request context, including its full
     * state (e.g. authenticated user, etc.).
     *
     * @param callable the runnable to wrap
     * @return the wrapped runnable
     * @since 7.9
     */
    @Nonnull
    <T> Callable<T> wrap(@Nonnull Callable<T> callable);

    /**
     * Wraps the provided {@link Runnable} so that it executes with this request context, including its full
     * state (e.g. authenticated user, etc.).
     *
     * @param runnable the runnable to wrap
     * @return the wrapped runnable
     * @since 7.9
     */
    @Nonnull
    Runnable wrap(@Nonnull Runnable runnable);
}
