package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.project.Project;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkState;

/**
 * Defines the properties that can be modified when updating a {@link Repository}. To reduce backwards compatibility
 * issues as new properties are added over time, instances of this class may only be created using its {@link Builder}.
 * <p>
 * The following properties are <i>required</i>:
 * <ul>
 *     <li>{@link #getId()}: The {@link Repository#getId() ID} of the repository to update. This value <i>will not be
 *     changed</i>.</li>
 *     <li>{@link #getName()}: The new name for the repository, which may match the old name if only the
 *     {@link #isForkable() forkability} of the repository should be updated</li>
 * </ul>
 * The following properties, if not specified, will have defaults applied:
 * <ul>
 *     <li>{@link #isForkable()}: Defaults to {@code true}, meaning the repository will be marked as forkable if not
 *     explicitly specified</li>
 *     <li>{@link #isPublic()}: Defaults to {@code false}, meaning the repository will not be public if not
 *     explicitly specified</li>
 * </ul>
 * The fork's {@link Repository#getSlug() slug}, which is used in URLs (both for the browser and when cloning), will be
 * generated from the provided {@link #getName()}. If the new name, or new slug, have already been used by a different
 * repository in the same {@link Repository#getProject() project}, the repository cannot be updated. Additionally, note
 * that changing a repository's name, as it may change the slug, may change <i>every URL</i> for the repository, both
 * URLs used by the browser <i>and</i> the clone URL. As a result, changing a repository's name may be very disruptive
 * and should be carefully considered.
 *
 * @see RepositoryService#update(RepositoryUpdateRequest)
 */
public class RepositoryUpdateRequest extends AbstractRepositoryRequest {

    private final Boolean archived;
    private final String defaultBranch;
    private final int id;
    private final Project project;

    private RepositoryUpdateRequest(Builder builder) {
        super(builder);
        
        checkState(StringUtils.isNotBlank(getName()));

        archived = builder.archived;
        defaultBranch = builder.defaultBranch;
        id = builder.id;
        project = builder.project;
    }

    /**
     * Retrieves the {@link Repository#isArchived() archive state} to be set when the repository is updated. If no value
     * is provided, the archive state will not be updated.
     *
     * @return the archive state, which may be {@code null} to retain the existing archive state
     * @since 8.0
     */
    @Nullable
    public Boolean getArchived() {
        return archived;
    }

    /**
     * Retrieves the new default branch that should be set when the repository is updated. If no value is provided
     * the default branch will not be updated.
     *
     * @return the default branch name, which may be {@code null} to retain the existing default branch
     * @since 7.5
     */
    @Nullable
    public String getDefaultBranch() {
        return defaultBranch;
    }

    /**
     * Retrieves the {@link Repository#getId() ID} of the repository to update.
     *
     * @return the repository's ID
     */
    public int getId() {
        return id;
    }

    /**
     * Retrieves the {@link Repository#getProject() project} the repository should be moved to.
     *
     * @return the project to move the repository to. If {@code null},
     *         leave the repository in the existing {@link Repository#getProject() project}
     */
    @Nullable
    public Project getProject() {
        return project;
    }

    /**
     * Constructs {@link RepositoryUpdateRequest requests} for updating repositories.
     */
    public static class Builder extends AbstractBuilder<Builder> {

        private Boolean archived;
        private String defaultBranch;
        private int id;
        private Project project;

        public Builder() {
        }

        /**
         * Constructs a new {@code Builder} which will update the specified {@link Repository}. The following values
         * are copied:
         * <ul>
         *     <li>{@link Repository#getId()} {@code ->} {@link #id(int)}</li>
         *     <li>{@link Repository#getName()} {@code ->} {@link #name(String)}</li>
         *     <li>{@link Repository#isArchived()} {@code ->} {@link #archived(boolean)}</li>
         *     <li>{@link Repository#isForkable()} {@code ->} {@link #forkable(boolean)}</li>
         *     <li>{@link Repository#isPublic()} {@code ->} {@link #publiclyAccessible(boolean)}</li>
         * </ul>
         * The caller may then change the specific fields they wish to update before building the request. If the
         * request is built immediately, the repository <i>will not be updated in any way</i>.
         *
         * @param repository the repository to update, and from which all default values should be drawn
         */
        public Builder(@Nonnull Repository repository) {
            super(repository);

            archived = repository.isArchived();
            id = repository.getId();
        }

        /**
         * Constructs a new {@code Builder} which uses as its defaults all the values from the provided request.
         *
         * @param request the request to copy into the new builder
         */
        public Builder(@Nonnull RepositoryUpdateRequest request) {
            super(request);

            archived = request.getArchived();
            defaultBranch = request.getDefaultBranch();
            id = request.getId();
            project = request.getProject();
        }

        /**
         * Builds a {@link RepositoryUpdateRequest request} from the assembled values. 
         *
         * @return the built request
         */
        @Nonnull
        public RepositoryUpdateRequest build() {
            return new RepositoryUpdateRequest(this);
        }

        /**
         * Sets the {@link Repository#isArchived() archive state} of the repository when it is updated.
         *
         * @param value {@code true} if the repository has been archived, {@code false} otherwise.
         * @return {@code this}
         * @since 8.0
         */
        @Nonnull
        public Builder archived(boolean value) {
            archived = value;

            return self();
        }

        /**
         * Sets the new default branch name to be applied when the repository is updated. If a default branch name
         * is not specified it will not be updated.
         *
         * @param value the new default branch name, or {@code null} to retain the existing name
         * @return {@code this}
         * @since 7.5
         */
        @Nonnull
        public Builder defaultBranch(@Nullable String value) {
            defaultBranch = StringUtils.trimToNull(value);

            return self();
        }

        /**
         * Sets the {@link Repository#getId() ID} of the repository to be updated.
         * <p>
         * Note: This value is <i>required</i>. If the ID specified does not match an existing repository, or if this
         * method is not called prior to {@link #build() building}, an exception will be thrown <i>when attempting to
         * perform the update</i>.
         *
         * @param value the repository's ID
         * @return {@code this}
         */
        @Nonnull
        public Builder id(int value) {
            id = value;

            return self();
        }

        /**
         * Sets the {@link Repository#getProject() project} the repository should be moved to.
         *
         * @param value the project to move the repository to. If {@code null},
         *              leave the repository in the existing {@link Repository#getProject() project}
         * @return {@code this}
         */
        @Nonnull
        public Builder project(@Nullable Project value) {
            project = value;

            return self();
        }

        /**
         * @return {@code this}
         */
        @Override
        protected Builder self() {
            return this;
        }
    }
}
