package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Base class for {@link Repository}-related request classes.
 */
public abstract class AbstractRepositoryRequest {

    private final String description;
    private final boolean forkable;
    private final String name;
    private final boolean publiclyAccessible;

    protected AbstractRepositoryRequest(AbstractBuilder builder) {
        description = builder.description;
        forkable = builder.forkable;
        name = builder.name;
        publiclyAccessible = builder.publiclyAccessible;
    }

    /**
     * The description for the repository. How this is used depends on the request type:
     * <ul>
     *     <li>{@link RepositoryCreateRequest}: Sets the description for the new repository</li>
     *     <li>{@link RepositoryForkRequest}: Sets the description for the new fork</li>
     *     <li>{@link RepositoryUpdateRequest}: Updates the repository's description. </li>
     * </ul>
     *
     * @return the description for the repository
     *
     * @since 6.2
     */
    @Nullable
    public String getDescription() {
        return description;
    }

    /**
     * The name for the repository. How this is used depends on the request type:
     * <ul>
     *     <li>{@link RepositoryCreateRequest}: Sets the name (and slug) for the new repository</li>
     *     <li>{@link RepositoryForkRequest}: Sets the name (and slug) for the new fork</li>
     *     <li>{@link RepositoryUpdateRequest}: Updates the repository's name, <i>which may also change its slug</i>.
     *     Changing a repository's slug will update <i>all of its URLs</i>, including its clone URL</li>
     * </ul>
     * The name provided is used to generate the repository's {@link Repository#getSlug() slug} regardless of the
     * request type.
     *
     * @return the name for the repository
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * Retrieves a flag indicating whether the repository should be {@link Repository#isForkable() forkable}. How this
     * is used depends on the request type:
     * <ul>
     *     <li>{@link RepositoryCreateRequest}: Sets whether the new repository will be forkable</li>
     *     <li>{@link RepositoryForkRequest}: Sets whether the new fork itself be forkable</li>
     *     <li>{@link RepositoryUpdateRequest}: Updates whether an existing repository is forkable</li>
     * </ul>
     *
     * @return {@code true} if the repository should be forkable; otherwise, {@code false} to disable forking
     */
    public boolean isForkable() {
        return forkable;
    }

    /**
     * @return {@code true} if the project will be made publicly accessible, {@code false} otherwise.
     */
    public boolean isPublic() {
        return publiclyAccessible;
    }

    /**
     * Base class for creating builders for {@link Repository repository}-related requests.
     *
     * @param <B> The builder class, used to control the return type on builder methods to provide a fluent API
     */
    public abstract static class AbstractBuilder<B extends AbstractBuilder<B>> extends BuilderSupport {

        private String description;
        private boolean forkable;
        private String name;
        private boolean publiclyAccessible;

        protected AbstractBuilder() {
            forkable = true; //Repositories are forkable by default
        }

        protected AbstractBuilder(@Nonnull Repository repository) {
            description = requireNonNull(repository, "repository").getDescription();
            forkable = repository.isForkable();
            name = repository.getName();
            publiclyAccessible = repository.isPublic();
        }

        protected AbstractBuilder(@Nonnull AbstractRepositoryRequest request) {
            description = requireNonNull(request, "request").getDescription();
            forkable = request.isForkable();
            name = request.getName();
            publiclyAccessible = request.isPublic();
        }

        /**
         * Sets the description of the repository.
         *
         * @param value the description of the repository
         * @return {@code this}
         *
         * @since 6.2
         */
        @Nonnull
        public B description(@Nullable String value) {
            description = value;

            return self();
        }

        /**
         * Sets whether the request should mark the repository forkable or not. If this method is not called prior to
         * building the request, its value defaults to {@code true} and the repository will be forkable by default.
         *
         * @param value {@code true} to mark the repository as forkable, or {@code false} to disable forking
         * @return {@code this}
         */
        @Nonnull
        public B forkable(boolean value) {
            forkable = value;

            return self();
        }

        /**
         * Sets the name to be used for the request. The provided value may not be {@code null}, empty or contain only
         * whitespace or an exception will be thrown.
         * <p>
         * Note: This value is <i>required</i>. If this method is not called prior to building the request, an
         * exception will be thrown.
         *
         * @param value the name for the repository
         * @return {@code this}
         * @throws IllegalArgumentException if the provided {@code value} is empty or contains only whitespace
         * @throws NullPointerException if the provided {@code value} is {@code null}
         */
        @Nonnull
        public B name(@Nonnull String value) {
            name = checkNotBlank(value, "name");

            return self();
        }

        /**
         * @param value whether the repository will be publicly accessible or not.
         * @return {@code this}
         */
        @Nonnull
        public B publiclyAccessible(boolean value) {
            publiclyAccessible = value;

            return self();
        }

        /**
         * Overridden in concrete builder implementations to return {@code this}.
         *
         * @return {@code this}
         */
        protected abstract B self();
    }
}
