package com.atlassian.bitbucket.pull.reviewer;

import com.atlassian.bitbucket.scope.Scope;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * @since 7.13
 * Request object used for {@link ReviewerGroupService#create(ReviewerGroupCreateRequest)} creating a reviewer group.
 */
public class ReviewerGroupCreateRequest {

    private final String description;
    private final String name;
    private final Scope scope;
    private final Set<ApplicationUser> users;

    private ReviewerGroupCreateRequest(Builder builder) {
        description = builder.description;
        name = builder.name;
        scope = builder.scope;
        users = builder.users.build();
    }

    /**
     * @return the description for a reviewer group
     */
    @Nullable
    public String getDescription() {
        return description;
    }

    /**
     * @return the name for a reviewer group
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * @return the scope of the reviewer group. A reviewer group can only be in repository and normal project
     *         scope level.
     */
    @Nonnull
    public Scope getScope() {
        return scope;
    }

    /**
     * @return the set of reviewers for the group. A reviewer group must contain at least one user. Passing
     *         an empty set to {@link ReviewerGroupService#create(ReviewerGroupCreateRequest)} will cause an
     *         {@link EmptyReviewerGroupException} to be thrown.
     */
    @Nonnull
    public Set<ApplicationUser> getUsers() {
        return users;
    }

    public static class Builder extends BuilderSupport {

        private final String name;
        private final Scope scope;
        private final ImmutableSet.Builder<ApplicationUser> users;

        private String description;

        public Builder(@Nonnull String name, @Nonnull Scope scope) {
            this.name = requireNonBlank(name, "name");
            this.scope = requireNonNull(scope, "scope");
            users = ImmutableSet.builder();
        }

        @Nonnull
        public ReviewerGroupCreateRequest build() {
            return new ReviewerGroupCreateRequest(this);
        }

        @Nonnull
        public Builder description(@Nullable String value) {
            description = value;

            return this;
        }

        @Nonnull
        public Builder users(@Nonnull ApplicationUser... values) {
            users.addAll(Arrays.asList(values));

            return this;
        }

        @Nonnull
        public Builder users(@Nonnull Iterable<ApplicationUser> values) {
            addIf(Objects::nonNull, users, values);

            return this;
        }
    }
}
