package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Base class for requests affecting pull requests.
 */
public class AbstractPullRequestRequest {

    private final long pullRequestId;
    private final int repositoryId;

    /**
     * @param builder a builder specifying the repository and pull request ID
     */
    protected AbstractPullRequestRequest(@Nonnull AbstractBuilder<?> builder) {
        pullRequestId = builder.pullRequestId;
        repositoryId = builder.repositoryId;
    }

    /**
     * @return the ID of the pull request within the repository
     */
    public long getPullRequestId() {
        return pullRequestId;
    }

    /**
     * @return the ID of the repository to which the pull request will be merged
     */
    public int getRepositoryId() {
        return repositoryId;
    }

    public abstract static class AbstractBuilder<T extends AbstractBuilder<T>> extends BuilderSupport {

        private final long pullRequestId;
        private final int repositoryId;

        protected AbstractBuilder(@Nonnull PullRequest pullRequest) {
            this(requireNonNull(pullRequest, "pullRequest").getToRef().getRepository().getId(), pullRequest.getId());
        }

        protected AbstractBuilder(int repositoryId, long pullRequestId) {
            this.repositoryId = repositoryId;
            this.pullRequestId = pullRequestId;
        }

        @Nonnull
        protected abstract T self();
    }
}
