package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Objects;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @since 8.0
 */
public class MeshMigrationRequest {

    public static final long MIN_BYTES_PER_SECOND = 131072L; // 1Mbit/s

    private final boolean all;
    private final long maxBytesPerSecond;
    private final Set<Integer> projectIds;
    private final Set<Integer> repositoryIds;

    private MeshMigrationRequest(@Nonnull Builder builder) {
        all = builder.all;
        maxBytesPerSecond = builder.maxBytesPerSecond;
        projectIds = builder.projectIds.build();
        repositoryIds = builder.repositoryIds.build();
    }

    /**
     * @return the maximum network bandwidth that should be consumed by migration
     * @since 7.18
     */
    public long getMaxBytesPerSecond() {
        return maxBytesPerSecond;
    }

    /**
     * @return the IDs of projects to be migrated
     * @since 8.0
     */
    @Nonnull
    public Set<Integer> getProjectIds() {
        return projectIds;
    }

    /**
     * @return the IDs of repositories to be migrated
     * @since 8.0
     */
    @Nonnull
    public Set<Integer> getRepositoryIds() {
        return repositoryIds;
    }

    /**
     * @return {@code true} if all repositories in the system are to be migrated
     * @since 8.0
     */
    public boolean isAll() {
        return all;
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableSet.Builder<Integer> projectIds;
        private final ImmutableSet.Builder<Integer> repositoryIds;

        private boolean all;
        private long maxBytesPerSecond;

        public Builder() {
            maxBytesPerSecond = Long.MAX_VALUE;
            projectIds = new ImmutableSet.Builder<>();
            repositoryIds = new ImmutableSet.Builder<>();
        }

        @Nonnull
        public Builder all(boolean value) {
            all = value;
            return this;
        }

        @Nonnull
        public MeshMigrationRequest build() {
            return new MeshMigrationRequest(this);
        }

        @Nonnull
        public Builder maxBytesPerSecond(long value) {
            checkArgument(value > MIN_BYTES_PER_SECOND, "maxBytesPerSecond must be at least " + MIN_BYTES_PER_SECOND);
            maxBytesPerSecond = value;
            return this;
        }

        @Nonnull
        public Builder projectId(int value) {
            projectIds.add(value);
            return this;
        }

        @Nonnull
        public Builder projectIds(@Nonnull Iterable<Integer> values) {
            addIf(Objects::nonNull, projectIds, values);
            return this;
        }

        @Nonnull
        public Builder repositoryId(int value) {
            repositoryIds.add(value);
            return this;
        }

        @Nonnull
        public Builder repositoryIds(@Nonnull Iterable<Integer> values) {
            addIf(Objects::nonNull, repositoryIds, values);
            return this;
        }
    }
}
