package com.atlassian.bitbucket.job;

import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import java.util.Optional;

/**
 * A service for managing {@link Job jobs}.
 *
 * @since 5.13
 */
public interface JobService {

    /**
     * Counts all messages matching the supplied request object
     * @param request the request
     * @return the number of matching messages
     */
    long countMessages(@Nonnull JobMessageSearchRequest request);

    /**
     * Creates a new {@link Job} from the supplied request object
     * @param request the request
     * @return the new {@link Job}
     */
    @Nonnull
    Job create(@Nonnull JobCreationRequest request);

    /**
     * Creates a new {@link JobMessage} from the supplied request object
     * @param request the request
     * @return the new {@link JobMessage}
     */
    @Nonnull
    JobMessage createMessage(@Nonnull JobMessageCreationRequest request);

    /**
     * Finds all {@link Job jobs} matching the supplied request object
     * @param request the request
     * @param pageRequest the page request
     * @return the page of matching {@link Job jobs}
     */
    @Nonnull
    Page<Job> search(@Nonnull JobSearchRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Finds all {@link JobMessage job messages} matching the supplied request object
     * @param request the request
     * @param pageRequest the page request
     * @return the page of matching {@link JobMessage job messages}
     */
    @Nonnull
    Page<JobMessage> searchMessages(@Nonnull JobMessageSearchRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Gets a {@link Job} by its ID
     * @param jobId the ID of the job
     * @return the matching Job or null if no such {@link Job} exists
     */
    @Nonnull
    Optional<Job> getById(long jobId);

    /**
     * Updates a {@link Job}
     * @param request the update request
     * @return the updated job
     */
    @Nonnull
    Job update(@Nonnull JobUpdateRequest request);
}