package com.atlassian.bitbucket.job;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * A request to find {@link Job jobs}
 *
 * @since 5.13
 */
public class JobSearchRequest {

    private final Set<JobState> states;
    private final String type;

    private JobSearchRequest(Builder builder) {
        states = builder.states.build();
        type = builder.type;
    }

    @Nonnull
    public Set<JobState> getStates() {
        return states;
    }

    @Nonnull
    public Optional<String> getType() {
        return Optional.ofNullable(type);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        JobSearchRequest that = (JobSearchRequest) o;
        return Objects.equals(getType(), that.getType());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getType());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("type", type)
                .toString();
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableSet.Builder<JobState> states;
        
        private String type;

        public Builder() {
            states = ImmutableSet.builder();
        }

        @Nonnull
        public JobSearchRequest build() {
            return new JobSearchRequest(this);
        }

        @Nonnull
        public Builder state(@Nonnull JobState value) {
            addIf(Objects::nonNull, states, value);
            return this;
        }

        @Nonnull
        public Builder states(@Nonnull JobState value, @Nonnull JobState... moreValues) {
            addIf(Objects::nonNull, states, value, moreValues);
            return this;
        }

        @Nonnull
        public Builder states(@Nonnull Iterable<JobState> values) {
            addIf(Objects::nonNull, states, values);
            return this;
        }

        @Nonnull
        public Builder type(@Nullable String value) {
            type = trimToNull(value);
            return this;
        }
    }
}
