package com.atlassian.bitbucket.hook.script;

import com.atlassian.bitbucket.hook.repository.RepositoryHookTrigger;
import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.scope.Scope;
import com.atlassian.bitbucket.scope.ScopeType;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import java.io.InputStream;
import java.util.Optional;

/**
 * A service for creating, updating and configuring {@link HookScript hook scripts}.
 * <p>
 * An SPI is provided which allows app developers to write hooks in Java, but for many that is not their language of
 * choice. Enter hook scripts. Hook scripts allow hooks to be written in other languages, such as Python or Perl, or
 * as shell scripts. App developers can create hook scripts, providing their contents, and then configure the scopes
 * in which they should run (optionally filtered to specific {@link RepositoryHookTrigger triggers}).
 *
 * @since 6.2
 */
public interface HookScriptService {

    /**
     * Creates a new hook script.
     * <p>
     * When creating a hook script, apps are required to provide the contents for the script, as well as a name and
     * their plugin key.
     * <ul>
     *     <li>The contents are {@link #getMaxSize limited} for disk space, but are otherwise not constrained.</li>
     *     <li>The name is free-form, and limited to 255 characters. Operating system-specific limitations to
     *     filenames <i>do not apply</i> to hook script names.
     *     <ul>
     *         <li><b>Note</b>: Uniqueness is <i>not enforced</i> for hook script names, so apps can create multiple
     *         hooks with the same name</li>
     *     </ul>
     *     </li>
     *     <li>The plugin key, also limited to 255 characters, correlates all of the scripts created by a given app,
     *     allowing them to be {@link #findByPluginKey found} later.</li>
     * </ul>
     * <p>
     * When a script is created, it is assigned a <i>globally unique</i> {@link HookScript#getId ID}. <i>App developers
     * are required to track the IDs of the scripts they create</i>, to allow them to manage them later. Scripts can
     * potentially be {@link #findByPluginKey found again using the plugin key}, but {@link #getById IDs} are intended
     * to be the primary mechanism for {@link #setConfiguration configuring}, {@link #delete deleting} or {@link #update
     * updating} scripts after they're created.
     * <p>
     * Creating a hook script requires {@link Permission#SYS_ADMIN SYS_ADMIN permission}.
     *
     * @param request a request describing the script to create
     * @return the created script, whose {@link HookScript#getId ID} should be retained by the creating app
     */
    @Nonnull
    HookScript create(@Nonnull HookScriptCreateRequest request);

    /**
     * Deletes the specified hook script, and removes any configuration which has been applied.
     * <p>
     * Deleting a hook script requires {@link Permission#SYS_ADMIN SYS_ADMIN permission}.
     *
     * @param script the script to delete
     */
    void delete(@Nonnull HookScript script);

    /**
     * Deletes all of the hook scripts associated with the specified plugin key, and removes any configuration which
     * has been applied for them.
     * <p>
     * Deleting hook scripts requires {@link Permission#SYS_ADMIN SYS_ADMIN permission}.
     * <p>
     * <b>Note</b>: When bulk deleting hook scripts, <i>no event is raised</i>. If {@link HookScriptDeletedEvent}s
     * are important, callers should {@link #findByPluginKey find the hook scripts} and {@link #delete delete them}
     * individually, which will trigger an event for each.
     *
     * @param pluginKey the plugin key to delete all hook scripts for
     * @return the number of hook scripts that were deleted
     */
    int deleteByPluginKey(@Nonnull String pluginKey);

    /**
     * Finds a hook script by its globally unique {@link HookScript#getId ID}, which was assigned when the script was
     * {@link #create created}.
     * <p>
     * If the caller expects the script to exist {@link #getById} can be used instead to drop the {@code Optional}
     * wrapping, but an exception will be thrown if the script doesn't exist.
     * <p>
     * Retrieving a hook script does not require any specific permission.
     *
     * @param scriptId the {@link HookScript#getId ID} of the script to retrieve
     * @return the script with the specified ID, or {@code empty()} if no script exists with that ID
     * @see #getById(long)
     */
    @Nonnull
    Optional<HookScript> findById(long scriptId);

    /**
     * Finds a page of hook scripts which were created by the app with the specified plugin key.
     * <p>
     * Retrieving hook scripts does not require any specific permission.
     *
     * @param pluginKey   the plugin key to search for
     * @param pageRequest describes the page of scripts to return
     * @return a page of scripts, which may be empty but never {@code null}
     */
    @Nonnull
    Page<HookScript> findByPluginKey(@Nonnull String pluginKey, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a hook script by its globally unique {@link HookScript#getId ID}, which was assigned when the script
     * was {@link #create created}.
     * <p>
     * If no script exists with the specified ID, an exception is thrown. If the caller anticipates that the script
     * may not exist, {@link #findById} can be used instead to avoid the exception.
     * <p>
     * Retrieving a hook script does not require any specific permission.
     *
     * @param scriptId the {@link HookScript#getId ID} of the script to retrieve
     * @return the script with the specified ID
     * @throws NoSuchHookScriptException if no script exists with the specified ID
     * @see #findById(long)
     */
    HookScript getById(long scriptId);

    /**
     * Retrieves the configured limit for hook script contents. Attempting to {@link #create} or {@link #update} a
     * script with larger contents will fail.
     *
     * @return the configured limit, in bytes
     */
    int getMaxSize();

    /**
     * Returns an {@link InputSupplier} which can be used to open an {@code InputStream} to read the <i>current</i>
     * contents for the specified hook script.
     * <p>
     * While hook scripts can be {@link #findById retrieved} by anyone, <i>reading</i> their contents requires
     * {@link Permission#SYS_ADMIN SYS_ADMIN} permission. Hook script <i>metadata</i>, which is provided by the
     * {@link HookScript} interface, is not sensitive, but hook script <i>contents</i> may contain credentials
     * (in order to access remote systems, for example) or other sensitive data.
     *
     * @param script the hook script to read
     * @return a supplier which can be used to stream the hook script's contents
     */
    @Nonnull
    InputSupplier<InputStream> read(@Nonnull HookScript script);

    /**
     * Removes any configuration for a given hook script from the specified {@link Scope scope}.
     * <p>
     * Removing a hook script's configuration means it <i>may</i> no longer run. If the script is also configured in
     * an overlapping, more general scope, that configuration will take over. For example, if a script is configured
     * for a specific repository and for the project which contains that repository, removing the repository-level
     * configuration will <i>not</i> prevent the script from running; the project-level configuration will be used
     * instead.
     * <p>
     * Removing configuration for a hook script requires the appropriate admin permission for the scope:
     * <ul>
     *     <li>{@link ScopeType#GLOBAL Global scope} requires {@link Permission#ADMIN ADMIN permission}</li>
     *     <li>{@link ScopeType#PROJECT Project scope} requires
     *     {@link Permission#PROJECT_ADMIN PROJECT_ADMIN permission}</li>
     *     <li>{@link ScopeType#REPOSITORY Repository scope} requires
     *     {@link Permission#REPO_ADMIN REPO_ADMIN permission}</li>
     * </ul>
     *
     * @param request a request describing which script to remove configuration for, and which scope to remove it from
     * @return {@code true} if configuration was removed; otherwise, {@code false} if the script was already not
     *         configured in the specified scope
     */
    boolean removeConfiguration(@Nonnull HookScriptRemoveConfigurationRequest request);

    /**
     * Configures a hook script to run for a given {@link Scope scope}. All scope types are supported, allowing scripts
     * to be configured to run for an individual repository, all repositories in a project, or all repositories. If the
     * script is already configured for the specified scope the existing configuration is <i>replaced</i>, including its
     * {@link HookScriptConfig#getTriggerIds triggers} (triggers are not cumulative).
     * <p>
     * If no {@link HookScriptSetConfigurationRequest#getTriggerIds trigger IDs} are specified, the script will run for
     * <i>all triggers</i> in the configured scope. Otherwise, it will only run for the specified triggers.
     * <p>
     * When scripts are run, the configuration for the <i>most specific scope</i> is used. Any configuration applied to
     * less-specific scopes is <i>ignored</i>; the levels do not merge. For example, if a script is configured for a
     * specific repository, as well as for the project the repository is in, when the script runs in that repository
     * it will use the repository-level configuration and the project-level configuration will be ignored.
     * <p>
     * Configuring a hook script requires the appropriate admin permission for the scope:
     * <ul>
     *     <li>{@link ScopeType#GLOBAL Global scope} -&gt; {@link Permission#ADMIN}</li>
     *     <li>{@link ScopeType#PROJECT Project scope} -&gt; {@link Permission#PROJECT_ADMIN}</li>
     *     <li>{@link ScopeType#REPOSITORY Repository scope} -&gt; {@link Permission#REPO_ADMIN}</li>
     * </ul>
     *
     * @param request a request describing which script to configure, and how
     * @return the configuration applied for the script
     */
    @Nonnull
    HookScriptConfig setConfiguration(@Nonnull HookScriptSetConfigurationRequest request);

    /**
     * Updates a hook script.
     * <p>
     * Hook scripts support updates to their:
     * <ul>
     *     <li>Contents</li>
     *     <li>{@link HookScript#getName Name}</li>
     *     <li>{@link HookScript#getPluginKey Plugin key}</li>
     * </ul>
     * Requests can update any combination of those properties at once. Updates to <i>any</i> of those properties will
     * update the script's {@link HookScript#getUpdatedDate last update date}, but <i>only</i> updating the script's
     * contents will update its {@link HookScript#getVersion version}.
     * <p>
     * Updating a hook script requires {@link Permission#SYS_ADMIN SYS_ADMIN permission}.
     *
     * @param request a request describing the hook script to update, and how
     * @return the updated script
     */
    @Nonnull
    HookScript update(@Nonnull HookScriptUpdateRequest request);

    /**
     * Updates existing hook scripts which reference {@code oldPluginKey} to reference {@code newPluginKey}.
     * <p>
     * Even configured hook scripts are only executed as long as the app associated with their plugin key is still
     * installed and enabled. That means if an app's plugin key changes, any hook scripts it created will no longer
     * execute until they are updated to reference its new plugin key. This method facilitates that update.
     * <p>
     * Updating hook script plugin keys requires {@link Permission#SYS_ADMIN SYS_ADMIN permission}. Additionally,
     * it is required that no installed app, enabled or disabled, have the specified {@code oldPluginKey}, and the
     * {@code newPluginKey} must reference an installed <i>and enabled</i> app.
     * <p>
     * <b>Warning</b>: <i>No "duplicate" checking is performed when updating plugin keys.</i> If an app creates a
     * hook script with a given plugin key, then its plugin key is changed and it creates a new hook script with
     * the new plugin key, calling this method will result in <i>two</i> hook scripts that use the new key, even
     * if the two hook scripts have the same {@link HookScript#getName name}.
     *
     * @param oldPluginKey the app's old plugin key
     * @param newPluginKey the app's new plugin key
     * @return the number of hook scripts updated with a new plugin key
     */
    int updatePluginKey(@Nonnull String oldPluginKey, @Nonnull String newPluginKey);
}
