package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.hook.ScmHookDetails;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.repository.Repository;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * Basic implementation of {@link RepositoryHookRequest}
 *
 * @since 5.0
 */
public class SimpleRepositoryHookRequest extends AbstractRepositoryHookRequest {

    private final Collection<RefChange> refChanges;
    private final ScmHookDetails scmHookDetails;

    protected SimpleRepositoryHookRequest(AbstractBuilder<?> builder) {
        super(builder);

        refChanges = builder.refChanges.build();
        scmHookDetails = builder.scmHookDetails;
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return refChanges;
    }

    @Nonnull
    @Override
    public Optional<ScmHookDetails> getScmHookDetails() {
        return ofNullable(scmHookDetails);
    }

    public static class Builder extends AbstractBuilder<Builder> {

        public Builder(@Nonnull Repository repository, @Nonnull RepositoryHookTrigger trigger) {
            super(repository, trigger);
        }

        @Nonnull
        public SimpleRepositoryHookRequest build() {
            return new SimpleRepositoryHookRequest(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }

    protected abstract static class AbstractBuilder<B extends AbstractBuilder<B>> extends
            AbstractRepositoryHookRequest.AbstractBuilder<B> {

        private final ImmutableSet.Builder<RefChange> refChanges;

        private ScmHookDetails scmHookDetails;

        protected AbstractBuilder(@Nonnull Repository repository, @Nonnull RepositoryHookTrigger trigger) {
            super(repository, trigger);

            refChanges = ImmutableSet.builder();
        }

        @Nonnull
        public B refChanges(@Nullable RefChange value, @Nullable RefChange...moreValues) {
            addIf(Objects::nonNull, refChanges, value, moreValues);
            return self();
        }

        @Nonnull
        public B refChanges(@Nullable Iterable<RefChange> values) {
            addIf(Objects::nonNull, refChanges, values);
            return self();
        }

        @Nonnull
        public B scmHookDetails(@Nullable ScmHookDetails value) {
            scmHookDetails = value;
            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
