package com.atlassian.bitbucket.event.tag;

import com.atlassian.bitbucket.hook.repository.RepositoryHookRequest;
import com.atlassian.bitbucket.hook.repository.StandardRepositoryHookTrigger;
import com.atlassian.bitbucket.repository.*;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Collections;

/**
 * A {@link RepositoryHookRequest hook request} that is raised just before a tag is deleted using the UI or REST.
 * Tags deleted by pushing will trigger a standard {@link RepositoryHookRequest} and not this specific type.
 *
 * @see StandardRepositoryHookTrigger#TAG_DELETE
 * @since 5.0
 */
public class TagDeletionHookRequest extends TagHookRequest {

    private final Collection<RefChange> changes;

    private TagDeletionHookRequest(Builder builder) {
        super(builder);

        changes = Collections.singleton(new SimpleRefChange.Builder()
                .from(getTag())
                .toHash(StringUtils.repeat("0", 40))
                .type(RefChangeType.DELETE)
                .build());
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return changes;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        public Builder(@Nonnull Repository repository, @Nonnull Tag tag) {
            super(repository, StandardRepositoryHookTrigger.TAG_DELETE, tag);
        }

        @Nonnull
        public TagDeletionHookRequest build() {
            return new TagDeletionHookRequest(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
