package com.atlassian.bitbucket.event.repository;

import com.atlassian.bitbucket.event.pull.PullRequestMergedEvent;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;

/**
 * Interface for events that are published when changes are made to a repository.
 * <p>
 * {@link RepositoryRefsChangedEvent} listeners will be notified when any
 * sub-events are published.  As a result, listening for this event is the
 * best way to be aware of repository changes.
 */
public interface RepositoryRefsChangedEvent {

    /**
     * Retrieves a collection of the {@link RefChange changed refs}.
     * <p>
     * <b>Note</b>: While the returned collection will never be {@code null}, <i>it may be empty</i>. For example, a
     * {@link PullRequestMergedEvent#isMergedRemotely() remotely merged pull request}
     * will have an empty set of changes.
     *
     * @return a collection containing 0 or more changed refs
     */
    @Nonnull
    Collection<RefChange> getRefChanges();

    /**
     * Retrieves the repository in which refs were changed. All of the {@link #getRefChanges() changed refs} for a
     * single event will <i>always</i> be in the same repository.
     *
     * @return the repository containing the changed refs
     */
    @Nonnull
    Repository getRepository();

    /**
     * Retrieves the user who triggered the ref changes, if available.
     * <p>
     * In most contexts, the user will be set. However, some changes may happen in contexts where no context user is
     * available. For example, custom plugins may perform automated processing outside of a user context and raise
     * change events based on the processing performed. Callers are encouraged to check for and handle the case where
     * there is no user available.
     *
     * @return the triggering user, or {@code null} if the change happened due to background processing or in another
     *         context where no user is available
     */
    @Nullable
    ApplicationUser getUser();
}
