package com.atlassian.bitbucket.event.repository;

import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestParticipantRequest;
import com.atlassian.bitbucket.pull.PullRequestRole;
import com.atlassian.bitbucket.pull.PullRequestState;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * Event that is raised when {@link PullRequest}s are filtered by one or more criteria from the UI.
 * <p>
 * This event is internally audited
 * 
 * @since 7.3
 */
@AsynchronousPreferred
public class PullRequestFilterEvent extends RepositoryEvent {

    private final String branch;
    private final String filterText;
    private final PullRequestState state;

    private String author;
    private String reviewer;

    public PullRequestFilterEvent(@Nonnull Object source, @Nonnull Repository repository,
                                  @Nullable PullRequestState state,
                                  @Nonnull Iterable<PullRequestParticipantRequest> prParticipants,
                                  @Nullable String branch,
                                  @Nullable String filterText) {
        super(source, repository);

        this.branch = branch;
        this.filterText = filterText;
        this.state = state;

        requireNonNull(prParticipants, "prParticipants");
        setAuthorAndReviewer(prParticipants);
    }

    /**
     * @return optional author used for filtering {@link PullRequest}s
     */
    @Nonnull
    public Optional<String> getAuthor() {
        return Optional.ofNullable(author);
    }

    /**
     * @return optional branch used for filtering {@link PullRequest}s
     */
    @Nonnull
    public Optional<String> getBranch() {
        return Optional.ofNullable(branch);
    }

    /**
     * @return optional text used for filtering {@link PullRequest}s
     */
    @Nonnull
    public Optional<String> getFilterText() {
        return Optional.ofNullable(filterText);
    }

    /**
     * @return optional reviewer used for filtering {@link PullRequest}s
     */
    @Nonnull
    public Optional<String> getReviewer() {
        return Optional.ofNullable(reviewer);
    }

    /**
     * @return optional {@link PullRequestState} used for filtering {@link PullRequest}s
     */
    @Nonnull
    public Optional<PullRequestState> getState() {
        return Optional.ofNullable(state);
    }

    private void setAuthorAndReviewer(Iterable<PullRequestParticipantRequest> prParticipants) {
        prParticipants.forEach(participant -> {
            if (participant.getRole() == PullRequestRole.AUTHOR) {
                author = participant.getUsername();
            }

            if (participant.getRole() == PullRequestRole.REVIEWER) {
                reviewer = participant.getUsername();
            }
        });
    }
}
