package com.atlassian.bitbucket.event.pull;

import com.atlassian.bitbucket.comment.Comment;
import com.atlassian.bitbucket.comment.CommentAction;
import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.event.annotation.TransactionAware;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestAction;
import com.atlassian.bitbucket.util.CancelState;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Subclasses of this event are synchronous and allow callers to cancel the action and/or participate in the
 * transaction (by persisting related entities, for example).
 *
 * @since 6.6
 */
@TransactionAware(TransactionAware.When.IMMEDIATE) //Override PullRequestEvent's annotation so we're synchronous
public abstract class PullRequestCancelableCommentEvent extends PullRequestEvent implements CancelableEvent {

    private final CancelState cancelState;
    private final Comment comment;
    private final CommentAction commentAction;
    private final Comment parent;

    protected PullRequestCancelableCommentEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest,
                                                @Nonnull Comment comment, @Nullable Comment parent,
                                                @Nonnull CommentAction commentAction,
                                                @Nonnull CancelState cancelState) {
        super(source, pullRequest, PullRequestAction.COMMENTED);

        this.cancelState = requireNonNull(cancelState, "cancelState");
        this.comment = requireNonNull(comment, "comment");
        this.commentAction = requireNonNull(commentAction, "commentAction");
        this.parent = parent;
    }

    @Override
    public void cancel(@Nonnull KeyedMessage message) {
        cancelState.cancel(message);
    }

    @Nonnull
    public Comment getComment() {
        return comment;
    }

    @Nonnull
    public CommentAction getCommentAction() {
        return commentAction;
    }

    @Nullable
    public Comment getParent() {
        return parent;
    }

    @Override
    public boolean isCanceled() {
        return cancelState.isCanceled();
    }
}
