package com.atlassian.bitbucket.event.project;

import com.atlassian.bitbucket.project.Project;
import org.apache.commons.lang3.ObjectUtils;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Base class for constructing events which relate to project modifications.
 * <p>
 * {@link #getProject()} always returns the <i>current persisted state</i> for the project. Whether that matches
 * the {@link #getOldValue() old values} or the {@link #getNewValue() new values} depends on the specific event, and
 * will be documented on the derived event class.
 */
public class AbstractProjectModificationEvent extends ProjectEvent {

    protected final Project newValue;
    protected final Project oldValue;

    public AbstractProjectModificationEvent(@Nonnull Object source, @Nonnull Project project,
                                            @Nonnull Project oldValue, @Nonnull Project newValue) {
        super(source, project);

        this.newValue = requireNonNull(newValue, "newValue");
        this.oldValue = requireNonNull(oldValue, "oldValue");
    }

    @Nonnull
    public Project getNewValue() {
        return newValue;
    }

    @Nonnull
    public Project getOldValue() {
        return oldValue;
    }

    public boolean isDescriptionChanged() {
        return ObjectUtils.notEqual(getOldValue().getDescription(), getNewValue().getDescription());
    }

    public boolean isKeyChanged() {
        return ObjectUtils.notEqual(getOldValue().getKey(), getNewValue().getKey());
    }

    public boolean isNameChanged() {
        return ObjectUtils.notEqual(getOldValue().getName(), getNewValue().getName());
    }
}
