package com.atlassian.bitbucket.event.cluster;

import com.atlassian.bitbucket.cluster.ClusterInformation;
import com.atlassian.bitbucket.cluster.ClusterNode;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * Raised when a {@link #getAddedNode() new node} joins the cluster.
 * <p>
 * Due to the non-deterministic nature of event processing, it is possible the other nodes may have joined or left the
 * cluster. As a result, using the {@link ClusterInformation#getNodes() ClusterInformation}
 * to determine the cluster nodes may be inconsistent. {@link #getCurrentNodes()} is provided to simplify deterministic
 * processing in listeners.
 */
@AsynchronousPreferred
public class ClusterNodeAddedEvent extends ClusterMembershipEvent {

    private final boolean maybeNetworkPartionResolved;

    public ClusterNodeAddedEvent(@Nonnull Object source, @Nonnull ClusterNode node,
                                 @Nonnull Set<ClusterNode> currentNodes) {
        super(source, node, currentNodes);

        // if either this or the joining node has not fully started, it cannot have been a split brain
        maybeNetworkPartionResolved = node.isFullyStarted() &&
                currentNodes.stream().filter(ClusterNode::isLocal).anyMatch(ClusterNode::isFullyStarted);
    }

    /**
     * @return the added node, which <i>will</i> be in the {@link #getCurrentNodes() current nodes} set
     */
    @Nonnull
    public ClusterNode getAddedNode() {
        return getNode();
    }

    /**
     * @return {@code true} if the node joining the cluster may be the result of a network partition (split brain)
     *         having resolved, otherwise {@code false}.
     * @since 4.10
     */
    public boolean isMaybeNetworkPartitionResolved() {
        return maybeNetworkPartionResolved;
    }
}
