package com.atlassian.bitbucket.content;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

import javax.annotation.Nonnull;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.Math.min;

public class SimplePath implements Comparable<SimplePath>, Path, Serializable {

    public static final String[] EMPTY = new String[0];
    public static final SimplePath ROOT = new SimplePath(EMPTY);

    private static final Interner<String> INTERNER = Interners.newWeakInterner();
    private static final Pattern SLOSH_SPLIT_PATTERN = Pattern.compile("[/|\\\\]");
    private static final Pattern SPLIT_PATTERN = Pattern.compile("/");

    private final String[] path;

    private String stringRep;

    public SimplePath(Path path) {
        this(path instanceof SimplePath ? path.getComponents() : internAll(path.getComponents()));
    }

    public SimplePath(CharSequence path) {
        this.path = internAll(split(path));
    }

    public SimplePath(CharSequence path, boolean allowSloshes) {
        this.path = internAll(split(path, allowSloshes));
    }

    public SimplePath(SimplePath aParent, String aPath) {
        path = join(aParent.path, internAll(split(aPath)));
    }

    public SimplePath(SimplePath aParent, String... aPath) {
        path = join(aParent.path, internAll(aPath));
    }

    public SimplePath(SimplePath aParent, SimplePath aPath) {
        path = join(aParent.path, aPath.path);
    }

    public SimplePath(SimplePath aParent, String aPath, boolean allowSloshes) {
        path = join(aParent.path, internAll(split(aPath, allowSloshes)));
    }

    public SimplePath(String aParent, SimplePath aPath) {
        path = join(internAll(split(aParent)), aPath.path);
    }

    public SimplePath(List<String> start) {
        this(internAll(start.toArray(new String[start.size()])));
    }

    /*
     * Note, performance: private non-interning ctor, only use when the path is
     * already interned
     */
    private SimplePath(String... path) {
        this.path = path;
    }

    /**
     * Intelligently put a "/" between two strings.
     *
     * @param a first path component
     * @param b second path component
     * @return merged path
     */
    public static String join(String a, String b) {
        if (a == null) {
            a = "";
        }

        if (b == null) {
            b = "";
        }

        if (a.length() == 0) {
            return b;
        }

        if (b.length() == 0) {
            return a;
        }

        if (a.endsWith("/")) {
            a = a.substring(0, a.length() - 1);
        }

        if (b.startsWith("/")) {
            b = b.substring(1);
        }

        return a + "/" + b;
    }

    public static String[] split(CharSequence aPath) {
        return split(aPath, false);
    }

    public static String[] split(CharSequence aPath, boolean allowSloshes) {
        if ((aPath == null) || (aPath.length() == 0)) {
            return EMPTY;
        }
        if (((aPath.length() > 0) && (aPath.charAt(0) == '/'))) {
            aPath = aPath.subSequence(1, aPath.length()); // the split function
            // includes an extra
            // match, otherwise
        }
        if (aPath.length() == 0) {
            return EMPTY;
        }
        Pattern splitPattern = allowSloshes ? SLOSH_SPLIT_PATTERN : SPLIT_PATTERN;
        return splitPattern.split(aPath);
    }

    /**
     * Abbreviates the path by removing path components from the middle until the length of the {@link #getPath() path}
     * is not greater than {@code maxLength}.
     * <p>
     * The first and last path component are always in the returned path. If any path components are removed, an
     * ellipses ("...") will appear in the middle. The returned path may in some cases still be greater than the
     * specified {@code maxLength}.
     *
     * @param maxLength maximum number of characters
     * @return an abbreviated path
     */
    public SimplePath abbreviate(int maxLength) {
        return abbreviateImpl(maxLength).path;
    }

    /**
     * Returns a similar result to {@link #getPathComponents()}, but abbreviates
     * the result (inserts null at the break point)
     *
     * @param maxLength maximum length
     * @return an abbreviated array of path components for the current path
     */
    public SimplePath[] abbreviatePathComponents(int maxLength) {
        List<SimplePath> pc = abbreviateImpl(maxLength).pathComponents;
        return pc.toArray(new SimplePath[pc.size()]);
    }

    @Override
    public int compareTo(SimplePath o) {
        if (o == null) {
            return -1;
        }

        String[] path = o.path;
        int i;
        for (i = 0; (i < this.path.length) && (i < path.length); i++) {
            String a = this.path[i];
            String b = path[i];
            int r = a.compareTo(b);
            if (r != 0) {
                return r;
            }
        }
        // they have a common prefix up to i (and i is the length of one of
        // them)
        if (this.path.length < path.length) {
            return -1;
        }
        if (path.length < this.path.length) {
            return 1;
        }
        return 0;

    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof SimplePath)) {
            return false;
        }

        SimplePath path = (SimplePath) o;

        return Arrays.equals(this.path, path.path);
    }

    public SimplePath getCommonRoot(SimplePath other) {
        int limit = min(path.length, other.path.length);
        int i = 0;
        while ((i < limit) && path[i].equals(other.path[i])) {
            ++i;
        }

        if (i == 0) {
            return ROOT;
        }
        // performance: use private non-interning ctor
        return new SimplePath(copy(i));
    }

    public String getComponent(int i) {
        return path[i];
    }

    @Nonnull
    public String[] getComponents() {
        return copy(path.length);
    }

    public String getExtension() {
        String extn = null;
        if (path.length > 0) {
            String file = getComponent(path.length - 1);
            int dot = file.lastIndexOf(".");
            if ((dot != -1) && (dot != file.length() - 1)) {
                extn = file.substring(dot + 1);
            }
        }
        return extn;
    }

    @Nonnull
    public String getName() {
        if (path.length == 0) {
            return "";
        }
        return path[path.length - 1];
    }

    public int getNumComponents() {
        return path.length;
    }

    public String getParent() {
        SimplePath parent = getParentPath();
        return parent != null ? parent.toString() : null;
    }

    public SimplePath getParentPath() {
        if (path.length == 0) {
            return null;
        }
        String[] comp = new String[path.length - 1];
        System.arraycopy(path, 0, comp, 0, comp.length);
        return new SimplePath(comp);
    }

    /**
     * @return the string representation of the path
     */
    public String getPath() {
        if (stringRep == null) {
            StringBuilder buf = new StringBuilder();
            String sep = "";
            for (String path : this.path) {
                buf.append(sep).append(path);
                sep = "/";
            }
            stringRep = buf.toString();
        }
        return stringRep;
    }

    public String getPath(boolean caseSensitive) {
        String normalized;
        if (caseSensitive) {
            normalized = getPath();
        } else {
            normalized = getPath().toLowerCase(Locale.US);
        }
        return normalized;
    }

    public SimplePath[] getPathComponents() {
        SimplePath[] paths = new SimplePath[path.length + 1];
        for (int i = 0; i <= path.length; i++) {
            // performance: use private non-interning ctor
            paths[i] = new SimplePath(copy(i));
        }
        return paths;
    }

    public SimplePath getPathHead() {
        if (path.length == 0) {
            return this;
        }
        return new SimplePath(path[0]);
    }

    /**
     * if other is an exact tail of this, return the path leading up to that
     * tail otherwise return null
     *
     * @param other the other path
     * @param caseSensitive true if case sensitive
     * @return the stripped path, or {@code null} if the other path is not a tail of this path
     */
    public SimplePath getStripTail(SimplePath other, boolean caseSensitive) {
        if (path.length < other.path.length) {
            return null;
        }
        for (int i = 0; i < other.path.length; i++) {
            String component1 = path[path.length - i - 1];
            String component2 = other.path[other.path.length - i - 1];
            if (caseSensitive ? !component1.equals(component2) : !component1.equalsIgnoreCase(component2)) {
                return null;
            }
        }
        return getSubPath(path.length - other.path.length);
    }

    public SimplePath getSubPath(int depth) {
        return new SimplePath(copy(min(depth, path.length)));
    }

    /**
     * @return a String composed of all the path segments after the first
     * segment
     */
    public String getTail() {
        return trimFirst(1).getPath();
    }

    public SimplePath getTailPath() {
        return trimFirst(1);
    }

    /**
     * determines if this path begins with all of <code>p</code>
     *
     * @param p the prefix to check
     * @return true if it has the prefix path
     */
    public boolean hasPrefix(SimplePath p) {
        return hasPrefix(p, true);
    }

    public boolean hasPrefix(SimplePath p, boolean caseSensitive) {
        if (p == null) {
            return true;
        }

        if (p.getNumComponents() > getNumComponents()) {
            return false; // p is too long
        }

        for (int i = 0; i < p.path.length; i++) {
            String p1 = p.path[i];
            String p0 = path[i];
            if ((caseSensitive && !p1.equals(p0)) || !p1.equalsIgnoreCase(p0)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        int result = 0;
        for (String s : path) {
            result += s.hashCode();
        }
        return result;
    }

    public boolean isAncestor(SimplePath descendant) {
        String[] parent = path;
        String[] child = descendant.path;
        if (child.length <= parent.length) {
            return false;
        }
        for (int i = 0; i < parent.length; i++) {
            if (!parent[i].equals(child[i])) {
                return false;
            }
        }
        return true;
    }

    public boolean isRoot() {
        return (path.length == 0) || ((path.length == 1) && (path[0].length() == 0));
    }

    public SimplePath simplify() {
        List<String> simplify = simplify(path);
        // performance: no need to intern, already interned internal path
        return new SimplePath(simplify.toArray(new String[simplify.size()]));
    }

    @Nonnull
    @Override
    public String toString() {
        return getPath();
    }

    public SimplePath trimFirst() {
        return trimFirst(1);
    }

    /**
     * @param n number of components to trim
     * @return the trimmed path or same path if the length was 0
     */
    public SimplePath trimFirst(int n) {
        if (path.length == 0) {
            return this; // we are already empty
        }

        int newLen = path.length - n;
        if (newLen <= 0) {
            return ROOT;
        }
        String[] p = new String[newLen];
        System.arraycopy(path, n, p, 0, p.length);
        return new SimplePath(p);
    }

    public SimplePath trimLast() {
        if (path.length == 0) {
            return this;
        }
        if (path.length == 1) {
            return ROOT;
        }
        String[] p = new String[path.length - 1];
        System.arraycopy(path, 0, p, 0, p.length);
        return new SimplePath(p);
    }

    private static String[] internAll(String[] components) {
        for (int i = 0; i < components.length; i++) {
            components[i] = INTERNER.intern(components[i]);
        }
        return components;
    }

    private static String[] join(String[] aLeft, String[] aRight) {
        int left = aLeft.length;
        int right = aRight.length;
        String[] result = new String[left + right];
        System.arraycopy(aLeft, 0, result, 0, left);
        System.arraycopy(aRight, 0, result, left, right);
        return result;
    }

    private static List<String> simplify(String[] aPath) {
        List<String> path = newArrayList(aPath);
        for (int i = 0; i < path.size(); i++) {
            String s1 = path.get(i);
            if ("..".equals(s1) && (i > 0)) {
                String s0 = path.get(i - 1);
                if (!"..".equals(s0)) {
                    path.remove(i--); // remove ".."
                    path.remove(i--); // remove previous element
                }
            } else if (".".equals(s1)) {
                path.remove(i--);
            }
        }
        if ((path.size() == 1)) {
            // turn [""] into []
            String s0 = path.get(0);
            if (s0.length() == 0) {
                path.remove(0);
            }
        }
        return path;
    }

    private AbbrevResult abbreviateImpl(int maxLength) {
        String ellipses = "...";

        if (path.length <= 2) {
            return new AbbrevResult(this, Arrays.asList(getPathComponents()));
        }

        int len = getPath().length();
        if (len <= maxLength) {
            return new AbbrevResult(this, Arrays.asList(getPathComponents()));
        }

        List<String> components = newArrayList(path);
        List<SimplePath> pathComponents = newArrayList(getPathComponents());

        int insertEllipsesAt = components.size() / 2;
        while (len > maxLength) {
            if (components.size() <= 2) {
                // always leave one at either end
                break;
            }
            int midPoint = components.size() / 2;
            insertEllipsesAt = min(insertEllipsesAt, midPoint);
            String s = components.remove(midPoint);
            pathComponents.remove(midPoint);
            len -= s.length() + 1;
        }

        components.add(insertEllipsesAt, ellipses); // ellipses
        pathComponents.add(insertEllipsesAt, null);

        // performance: no need to intern as components come from already
        // interned SimplePath
        return new AbbrevResult(new SimplePath(components.toArray(new String[components.size()])), pathComponents);
    }

    private String[] copy(int i) {
        String[] comp = new String[i];
        System.arraycopy(path, 0, comp, 0, i);
        return comp;
    }

    private static class AbbrevResult {

        private SimplePath path;
        private List<SimplePath> pathComponents;

        public AbbrevResult(SimplePath path, List<SimplePath> pathComponents) {
            this.path = path;
            this.pathComponents = pathComponents;
        }
    }
}
