package com.atlassian.bitbucket.content;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * A minimal subset of a {@link Change} describing how a path was changed on one side of a {@link Conflict}. For some
 * {@link ChangeType types of changes}, a {@link #getSrcPath() source path} will also be included.
 * <p>
 * Note: The path in the repository for which the {@link Conflict} happened may be <i>either</i> the {@link #getPath()}
 * or the {@link #getSrcPath()}.
 */
public interface ConflictChange {

    /**
     * Retrieves the path that was {@link #getType() changed} or, for {@link ChangeType#COPY copies} and
     * {@link ChangeType#MOVE moves/renames}, the path after the change was applied.
     *
     * @return the changed path or, for {@link ChangeType#MOVE moves/renames}, the path after the change
     */
    @Nonnull
    Path getPath();

    /**
     * For certain {@link ChangeType types of changes}, retrieves the path prior to the change.
     * <p>
     * {@link ChangeType#COPY Copies} and {@link ChangeType#MOVE moves/renames} should <i>attempt</i> to provide a
     * source path. However, due to the vagaries of source control, it is possible they will be unable to. Callers
     * should not assume this path will be provided even for copies, moves or renames.
     *
     * @return the path prior to the change
     */
    @Nullable
    Path getSrcPath();

    /**
     * Retrieves the {@link ChangeType type of change} that was made on this side of the conflict.
     *
     * @return the change type
     */
    @Nonnull
    ChangeType getType();
}
