package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.Tag;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Describes a request to retrieve the commits that last modified the files in the specified
 * {@link #getPath() path}, from a given {@link #getCommitId() starting commit}. The result
 * should be similar to {@link CommitService#getCommit retrieving the latest commit} for each
 * file, but may be significantly more efficient since it can be done in a single request.
 *
 * @since 4.6
 */
public class LastModifiedRequest extends AbstractCommitRequest {

    private LastModifiedRequest(Builder builder) {
        super(builder);
    }

    /**
     * Retrieves the <i>starting commit</i> to use. This ID serves two purposes:
     * <ul>
     *     <li>The {@link #getPath() path}'s file listing will use this commit</li>
     *     <li>The process of searching for modifications will only consider ancestors of this commit</li>
     * </ul>
     * Typically this commit will be the tip of a {@link Branch} or {@link Tag}. Depending on which files
     * that commit directly modified, however, it's entirely possible <i>none</i> of the resulting last
     * modified commits will be this one.
     *
     * @return the commit to serve as the starting point when searching for modifications
     */
    @Nonnull
    @Override
    public String getCommitId() {
        return super.getCommitId();
    }

    /**
     * Retrieves the path where file modifications should be searched. Commits which modify <i>files</i>
     * directly below this path will be traversed to determine the most recent commit to modify each file.
     * {@code null}, {@code ""} and {@code /} all indicate the root directory.
     * <p>
     * For example, consider a repository with the following contents. Entries ending with {@code /} are
     * directories.
     * <pre>
     * src/
     *   main/
     *     java/
     *       Example.java
     *     resources/
     *       atlassian-plugin.xml
     * pom.xml
     * </pre>
     * A request for the root directory will return the commit that last modified {@code pom.xml}, the only
     * file in the root directory. A request for {@code src/main} will not return <i>any</i> modifications
     * because that path only contains subdirectories.
     * <p>
     * For performance reasons, <i>there is no way to use this request to recursively find the last modification
     * for files in subdirectories.</i> Each level must be requested individually. This aligns with how
     * {@link com.atlassian.bitbucket.content.ContentService#streamDirectory streaming directories} works
     * when the {@code recursive} flag is {@code false}.
     *
     * @return the path whose file modifications should be searched, or {@code null} to search for file
     *         modifications in the root directory
     */
    @Nullable
    @Override
    public String getPath() {
        return super.getPath();
    }

    public static class Builder extends AbstractCommitRequestBuilder<Builder> {

        public Builder(@Nonnull Repository repository, @Nonnull String commitId) {
            super(repository, commitId);
        }

        @Nonnull
        public LastModifiedRequest build() {
            return new LastModifiedRequest(this);
        }

        @Override
        protected Builder self() {
            return this;
        }
    }
}
