package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * Provides context about the {@link LastModifiedRequest request} when {@link LastModifiedCallback streaming}
 * modifications.
 *
 * @since 4.6
 */
public class LastModifiedContext {

    private final String commitId;
    private final String path;

    private LastModifiedContext(Builder builder) {
        commitId = builder.commitId;
        path = builder.path;
    }

    /**
     * @return the {@link LastModifiedRequest#getCommitId starting commit} that was used to retrieve the file list
     *         and anchor the history traversal
     */
    @Nonnull
    public String getCommitId() {
        return commitId;
    }

    /**
     * @return the {@link LastModifiedRequest#getPath path} from which files are being streamed; all
     *         {@link LastModifiedCallback#onFile files} will be relative to this path
     */
    @Nonnull
    public Optional<String> getPath() {
        return ofNullable(path);
    }

    public static class Builder extends BuilderSupport {

        private final String commitId;

        private String path;

        public Builder(@Nonnull String commitId) {
            this.commitId = requireNonBlank(commitId, "commitId");
        }

        @Nonnull
        public LastModifiedContext build() {
            return new LastModifiedContext(this);
        }

        @Nonnull
        public Builder path(@Nullable String value) {
            path = value;

            return this;
        }
    }
}
