package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Defines a request for determining the common ancestor of a list of commits.
 *
 * @since 5.0
 */
public class CommonAncestorRequest {

    private final Set<String> commitIds;
    private final Repository repository;
    private final Repository secondaryRepository;

    private CommonAncestorRequest(Builder builder) {
        commitIds = builder.commitIds.build();
        checkArgument(commitIds.size() > 0, "Must specify at least 1 commit ID");
        repository = builder.repository;
        secondaryRepository = builder.secondaryRepository;
    }

    /**
     * @return the IDs of the commits whose common ancestor should be determined
     */
    @Nonnull
    public Set<String> getCommitIds() {
        return commitIds;
    }

    /**
     * @return the repository to determine the common ancestor in
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * When determining the common ancestor of commits in different repositories, retrieves the <b>secondary</b>
     * repository.
     *
     * @return a secondary repository whose commits should be considered
     */
    @Nullable
    public Repository getSecondaryRepository() {
        return secondaryRepository;
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableSet.Builder<String> commitIds;
        private final Repository repository;

        private Repository secondaryRepository;

        public Builder(@Nonnull Repository repository) {
            this.repository = requireNonNull(repository, "repository");
            commitIds = ImmutableSet.builder();
        }

        @Nonnull
        public CommonAncestorRequest build() {
            return new CommonAncestorRequest(this);
        }

        @Nonnull
        public Builder commitId(@Nullable String value, @Nullable String...moreValues) {
            addIf(Objects::nonNull, commitIds, value, moreValues);

            return this;
        }

        @Nonnull
        public Builder commitId(@Nullable Iterable<String> values) {
            addIf(Objects::nonNull, commitIds, values);

            return this;
        }

        @Nonnull
        public Builder secondaryRepository(@Nullable Repository value) {
            secondaryRepository = value;

            return this;
        }
    }
}
