package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.content.Change;
import com.atlassian.bitbucket.content.DiffFileType;
import com.atlassian.bitbucket.content.DiffSegmentType;

import javax.annotation.Nonnull;
import java.util.Optional;

/**
 * The location in a diff a {@link CommentThread comment thread} is anchored to.
 *
 * @since 5.0
 */
public interface CommentThreadDiffAnchor {

    int NO_LINE = 0;

    /**
     * @return the diff type associated with this anchor
     */
    @Nonnull
    CommentThreadDiffAnchorType getDiffType();

    /**
     * File type this anchor is anchored to in a diff.
     * <p>
     * The following relationship with {@link #getLineType()} occurs:
     * <ul>
     *     <li>for {@link DiffSegmentType#REMOVED} this will always be {@link DiffFileType#FROM}</li>
     *     <li>for {@link DiffSegmentType#ADDED} this will always be {@link DiffFileType#TO}</li>
     *     <li>for {@link DiffSegmentType#CONTEXT} the anchor can be hooked to any of the diff files</li>
     * </ul>
     *
     * @return type of the file this anchor is anchored to in a diff. It will only be
     * {@link Optional#isPresent() present} if this is a {@link #isLineAnchor() line anchor}
     * @see DiffFileType
     */
    @Nonnull
    Optional<DiffFileType> getFileType();

    @Nonnull
    Optional<String> getFromHash();

    int getLine();

    /**
     * @return type of the line this anchor is anchored to. It will only be
     * {@link Optional#isPresent() present} if this is a {@link #isLineAnchor() line anchor}
     * @see DiffSegmentType
     */
    @Nonnull
    Optional<DiffSegmentType> getLineType();

    /**
     * @return the path to the contents this anchor points to
     */
    @Nonnull
    String getPath();

    /**
     * The original path at which the {@link #getPath() content pointed to} by this anchor originated, or
     * {@link Optional#empty()} if the change was not {@link Change#getSrcPath() one that produces a new path}.
     *
     * @return the original path of the contents referred to by this anchor
     */
    @Nonnull
    Optional<String> getSrcPath();

    @Nonnull
    String getToHash();

    /**
     * @return {@code true}, if this is a file comment anchor, i.e. an anchor associated with the whole file rather than
     * a particular line. {@code true} implies that {@link #isLineAnchor()} returns {@code false}.
     * @see #isLineAnchor()
     */
    boolean isFileAnchor();

    /**
     * @return {@code true}, if this is a line comment anchor, i.e. an anchor associated with a particular line in the
     * file. {@code true} implies that {@link #isFileAnchor()} returns {@code false}.
     * @see #isFileAnchor()
     */
    boolean isLineAnchor();

    /**
     * An orphaned anchor point is out-of-date in the current state of a diff. E.g. the anchor points to a
     * path that is no longer in scope of the pull request, because the changes to that file have been reverted on
     * the source branch.
     *
     * @return {@code true}, if the anchor is orphaned, {@code false} otherwise
     */
    boolean isOrphaned();
}
