package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

import static com.atlassian.bitbucket.comment.CommentThreadDiffAnchorState.ACTIVE;
import static java.util.Objects.requireNonNull;

/**
 * A request for searching for {@link Comment comments} and {@link CommentThread threads}.
 *
 * @since 5.0
 */
public class CommentSearchRequest {

    private final CommentThreadDiffAnchorState anchorState;
    private final ApplicationUser author;
    private final Commentable commentable;
    private final Set<CommentThreadDiffAnchorType> diffTypes;
    private final String fromHash;
    private final String path;
    private final ApplicationUser pendingAuthor;
    private final Set<CommentSeverity> severities;
    private final Set<CommentState> states;
    private final String toHash;

    private CommentSearchRequest(@Nonnull Builder builder) {
        anchorState = builder.anchorState;
        author = builder.author;
        commentable = builder.commentable;
        diffTypes = builder.diffTypes.build();
        fromHash = builder.fromHash;
        path = builder.path;
        pendingAuthor = builder.pendingAuthor;
        severities = builder.severities.build();
        states = builder.states.build();
        toHash = builder.toHash;
    }

    /**
     * @return the comment thread diff anchor state
     * @since 5.4
     */
    @Nonnull
    public CommentThreadDiffAnchorState getAnchorState() {
        return anchorState;
    }

    /**
     * @since 8.4
     */
    @Nullable
    public ApplicationUser getAuthor() {
        return author;
    }

    @Nonnull
    public Commentable getCommentable() {
        return commentable;
    }

    @Nonnull
    public Set<CommentThreadDiffAnchorType> getDiffTypes() {
        return diffTypes;
    }

    @Nullable
    public String getFromHash() {
        return fromHash;
    }

    @Nullable
    public String getPath() {
        return path;
    }

    /**
     * Limits {@link CommentState#PENDING pending} comments to those posted by the specified author. If
     * {@link #getStates()} doesn't include {@link CommentState#PENDING PENDING} this property will be
     * ignored, since no pending comments will be returned.
     *
     * @return the author to limit pending comments to, or {@code null} to not limit by author
     * @since 7.7
     */
    @Nullable
    public ApplicationUser getPendingAuthor() {
        return pendingAuthor;
    }

    /**
     * @since 6.7
     */
    @Nonnull
    public Set<CommentSeverity> getSeverities() {
        return severities;
    }

    /**
     * @since 6.7
     */
    @Nonnull
    public Set<CommentState> getStates() {
        return states;
    }

    @Nullable
    public String getToHash() {
        return toHash;
    }

    public static class Builder extends BuilderSupport {

        private final Commentable commentable;
        private final ImmutableSet.Builder<CommentThreadDiffAnchorType> diffTypes;
        private final ImmutableSet.Builder<CommentSeverity> severities;
        private final ImmutableSet.Builder<CommentState> states;

        private CommentThreadDiffAnchorState anchorState;
        private ApplicationUser author;
        private String fromHash;
        private String path;
        private ApplicationUser pendingAuthor;
        private String toHash;

        public Builder(@Nonnull Commentable commentable) {
            this.commentable = requireNonNull(commentable, "commentable");

            anchorState = ACTIVE;
            diffTypes = ImmutableSet.builder();
            severities = ImmutableSet.builder();
            states = ImmutableSet.builder();
        }

        /**
         * @param value the new anchor state
         * @return {@code this}
         * @since 5.4
         */
        @Nonnull
        public Builder anchorState(@Nonnull CommentThreadDiffAnchorState value) {
            anchorState = requireNonNull(value, "anchorState");
            return this;
        }

        /**
         * @since 8.4
         */
        @Nonnull
        public Builder author(@Nullable ApplicationUser value) {
            author = requireNonNull(value, "author");
            return this;
        }

        @Nonnull
        public CommentSearchRequest build() {
            return new CommentSearchRequest(this);
        }

        @Nonnull
        public Builder diffType(@Nullable CommentThreadDiffAnchorType value) {
            addIf(Objects::nonNull, diffTypes, value);
            return this;
        }

        @Nonnull
        public Builder diffTypes(@Nullable Iterable<CommentThreadDiffAnchorType> values) {
            addIf(Objects::nonNull, diffTypes, values);
            return this;
        }

        @Nonnull
        public Builder fromHash(@Nullable String value) {
            fromHash = value;
            return this;
        }

        @Nonnull
        public Builder path(@Nullable String value) {
            path = value;
            return this;
        }

        /**
         * @since 7.7
         */
        @Nonnull
        public Builder pendingAuthor(@Nullable ApplicationUser value) {
            pendingAuthor = value;
            return this;
        }

        /**
         * @since 6.7
         */
        @Nonnull
        public Builder severities(@Nullable Iterable<CommentSeverity> values) {
            addIf(Objects::nonNull, severities, values);
            return this;
        }

        /**
         * @since 6.7
         */
        @Nonnull
        public Builder severity(@Nullable CommentSeverity value) {
            addIf(Objects::nonNull, severities, value);
            return this;
        }

        /**
         * @since 6.7
         */
        @Nonnull
        public Builder state(@Nullable CommentState value) {
            addIf(Objects::nonNull, states, value);
            return this;
        }

        /**
         * @since 6.7
         */
        @Nonnull
        public Builder states(@Nullable Iterable<CommentState> values) {
            addIf(Objects::nonNull, states, values);
            return this;
        }

        @Nonnull
        public Builder toHash(@Nullable String value) {
            toHash = value;
            return this;
        }
    }
}
