package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.property.PropertySupport;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.validation.annotation.RequiredString;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Date;
import java.util.List;
import java.util.Optional;

/**
 * A comment on a pull request or commit.
 */
public interface Comment extends PropertySupport {

    /**
     * This is a short-hand for {@code comment.getThread().getAnchor()}
     *
     * @return the anchor associated with this comment's {@link CommentThread thread}
     * @since 5.0
     */
    @Nonnull
    Optional<CommentThreadDiffAnchor> getAnchor();

    /**
     * @return the author of the comment
     */
    @Nonnull
    ApplicationUser getAuthor();

    /**
     * @return the direct replies to the comment, or an empty list of there are none
     */
    @Nonnull
    List<Comment> getComments();

    /**
     * @return the date when the comment was originally created
     * @see #getUpdatedDate()
     */
    @Nonnull
    Date getCreatedDate();

    /**
     * @return the comment's ID, which is globally unique
     */
    long getId();

    /**
     * @return the operations allowed by the current user. If not {@link Optional#isPresent() present}
     *         then none are permitted
     * @since 5.0
     */
    @Nonnull
    Optional<CommentOperations> getPermittedOperations();

    /**
     * @return the date that the comment was resolved or null if not resolved.
     * @since 6.7
     */
    @Nullable
    Date getResolvedDate();

    /**
     * Retrieves the {@link ApplicationUser user} who resolved the comment.
     *
     * @return the user who resolved the comment or null if not resolved
     * @since 6.7
     */
    @Nullable
    ApplicationUser getResolver();

    /**
     * @return the {@link CommentSeverity severity} for this comment
     * @since 6.7
     */
    @Nonnull
    CommentSeverity getSeverity();

    /**
     * @return the {@link CommentState state} for this comment
     * @since 6.7
     */
    @Nonnull
    CommentState getState();

    /**
     * @return the raw text of the comment (limited to 32k)
     */
    @Nonnull
    @RequiredString(size = 32768) //32k
    String getText();

    /**
     * @return the thread this comment is a part of
     * @since 5.0
     */
    @Nonnull
    CommentThread getThread();

    /**
     * @return the date of the last update
     * @see #getCreatedDate()
     */
    @Nonnull
    Date getUpdatedDate();

    /**
     * @return the current version of the comment
     * @see #getUpdatedDate()
     */
    int getVersion();
}
