package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * A request for creating comments on a {@link Commentable}.
 *
 * @since 5.0
 */
public class AddCommentRequest {

    private final Commentable commentable;
    private final boolean pending;
    private final CommentSeverity severity;
    private final String text;

    protected AddCommentRequest(@Nonnull AbstractBuilder builder) {
        commentable = builder.commentable;
        pending = builder.pending;
        severity = builder.severity;
        text = builder.text;
    }

    @Nonnull
    public Commentable getCommentable() {
        return commentable;
    }

    /**
     * @since 6.7
     */
    @Nullable
    public CommentSeverity getSeverity() {
        return severity;
    }

    @Nonnull
    public String getText() {
        return text;
    }

    /**
     * @since 7.7
     */
    public boolean isPending() {
        return pending;
    }

    public static class Builder extends AbstractBuilder<Builder, AddCommentRequest> {

        public Builder(@Nonnull Commentable commentable, @Nonnull String text) {
            super(commentable, text);
        }

        @Nonnull
        @Override
        public AddCommentRequest build() {
            return new AddCommentRequest(this);
        }

        @Nonnull
        @Override
        public Builder self() {
            return this;
        }
    }

    public abstract static class AbstractBuilder<B extends AbstractBuilder<B, R>, R extends AddCommentRequest>
            extends BuilderSupport {

        private final Commentable commentable;
        private final String text;

        private boolean pending;
        private CommentSeverity severity;

        public AbstractBuilder(@Nonnull Commentable commentable, @Nonnull String text) {
            this.commentable = requireNonNull(commentable, "commentable");
            this.text = requireNonBlank(text, "text");
        }

        @Nonnull
        public abstract R build();

        /**
         * @since 7.7
         */
        @Nonnull
        public B pending(boolean value) {
            pending = value;
            return self();
        }

        /**
         * @since 6.7
         */
        @Nonnull
        public B severity(@Nullable CommentSeverity value) {
            severity = value;
            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
