package com.atlassian.bitbucket.auth;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.user.NoSuchUserException;

import javax.annotation.Nonnull;

/**
 * @since 5.5
 */
public interface AuthenticationService {

    /**
     * Attempts to authenticate with the specified token.
     * <p>
     * Note: this method only authenticates; it does not apply the authentication to the current request, nor does it
     * call any authentication success or failure handlers.
     *
     * @param token the token
     * @return the {@link ApplicationUser} representing the authenticated user
     * @throws AuthenticationException if the provided token is invalid
     * @since 7.19
     */
    @Nonnull
    Authentication authenticate(@Nonnull String token);

    /**
     * Attempts to authenticate the specified user given their password. Failed authentications will not count towards
     * the maximum number allowed before CAPTCHA is imposed for the supplied user.
     * <p>
     * Note: this method only authenticates; it does not apply the authentication to the current request, nor does it
     * call any authentication success or failure handlers.
     *
     * @param username the username
     * @param password the user's password
     * @return the {@link ApplicationUser} representing the authenticated user
     * @throws AuthenticationSystemException if a failure occurs in Crowd
     * @throws ExpiredPasswordAuthenticationException if the user's password has expired and must be changed
     * @throws InactiveUserAuthenticationException if the specified user is inactive
     * @throws IncorrectPasswordAuthenticationException if the provided password is incorrect
     * @throws NoSuchUserException if the specified user does not exist or their user details cannot be retrieved
     *
     * @see #clear()
     */
    @Nonnull
    Authentication authenticate(@Nonnull String username, @Nonnull String password);

    /**
     * Attempts to authenticate the specified user given their password. Failed authentications will count towards
     * the maximum number allowed before CAPTCHA is imposed for the supplied user. If the maximum number of allowed
     * failed logins has already been exceeded the method will throw {@link CaptchaRequiredAuthenticationException}
     * <p>
     * Note: this method only authenticates; it does not apply the authentication to the current request, nor does it
     * call any authentication success or failure handlers.
     *
     * @param username the username
     * @param password the user's password
     * @return the {@link Authentication} representing the authenticated user and additional properties
     * @throws AuthenticationSystemException if a failure occurs in one of the authentication handlers
     * @throws CaptchaRequiredAuthenticationException if CAPTCHA is required
     * @throws ExpiredPasswordAuthenticationException if the user's password has expired and must be changed
     * @throws InactiveUserAuthenticationException if the specified user is inactive
     * @throws IncorrectPasswordAuthenticationException if the provided password is incorrect
     * @throws NoSuchUserException if the specified user does not exist or their user details cannot be retrieved
     */
    @Nonnull
    Authentication authenticateWithCaptcha(@Nonnull String username, @Nonnull String password);

    /**
     * Clears the current authentication, if any, resulting in an anonymous context
     *
     * @return the previously active authentication
     */
    @Nonnull
    Authentication clear();

    /**
     * @return the authentication that's active on the current thread
     */
    @Nonnull
    Authentication get();

    /**
     * Sets the authentication for the current thread (and thereby request)
     *
     * @param authentication the authentication to set
     * @return the previously active authentication
     */
    @Nonnull
    Authentication set(@Nonnull Authentication authentication);
}
