package com.atlassian.bitbucket.attachment;

import com.atlassian.bitbucket.NoSuchObjectException;
import com.atlassian.bitbucket.ServiceException;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.validation.ArgumentValidationException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

/**
 * Handling of attachments and attachments metadata.
 *
 * @since 7.0
 */
public interface AttachmentService {

    /**
     * Deletes the attachment with the specified ID, if it exists.
     * <p>
     * Note: If no attachment exists with the specified ID, no exception is thrown.
     *
     * @param repository the repository to which the attachment belongs
     * @param id         the ID as long assigned when the attachment was saved
     * @return {@code true} if the specified attachment was deleted; otherwise {@code false} if it did not exist or
     *         if it could not be deleted
     * @throws AttachmentStoreException if no attachment exists
     */
    boolean delete(@Nonnull Repository repository, long id);

    /**
     * Deletes that attachment metadata with the specified ID, if it exists.
     *
     * @param repository   the repository to which the attachment will have metadata associated with
     * @param attachmentId the ID of the attachment
     * @return {@code true} if the deletion of the attachment metadata is successful; otherwise {@code false} if we
     *         cannot delete the attachment metadata
     */
    boolean deleteMetadata(@Nonnull Repository repository, long attachmentId);

    /**
     * Get the attachment that matches the ID that is supplied.
     *
     * @param attachmentId the ID of the attachment
     * @return the attachment matching the ID or null if the attachment cannot be found with the ID supplied
     */
    @Nullable
    Attachment getById(long attachmentId);

    /**
     * Retrieves the maximum size, in bytes, for uploaded attachments. Attachments uploaded that are larger than this
     * will be rejected by the server.
     *
     * @return the maximum size, in bytes, for uploaded attachments
     */
    long getMaxUploadSize();

    /**
     * Get the attachment metadata that is associated with the attachment metadata ID that is supplied.
     *
     * @param repository the repository to which the attachment metadata belongs
     * @param attachmentId the ID of the attachment
     * @return the attachment metadata object that matches the id
     * @throws NoSuchObjectException if the attachment metadata cannot be found
     */
    @Nonnull
    AttachmentMetadata getMetadata(@Nonnull Repository repository, long attachmentId);

    /**
     * Reads the attachment with the specified ID.
     *
     * @param repository the repository to which the attachment belongs
     * @param id         the ID assigned when the attachment was saved
     * @return a supplier from which the attachment can be streamed
     * @throws NoSuchObjectException if no attachment exists
     */
    @Nonnull
    AttachmentSupplier read(@Nonnull Repository repository, long id);

    /**
     * Saves the attachment contained in the provided {@link AttachmentSupplier supplier}, assigning a unique ID to
     * allow it to be accessed again.
     *
     * @param repository         the repository to which the attachment belongs
     * @param attachmentSupplier a supplier containing the attachment data to store
     * @return the attachment object representing the newly created attachment
     * @throws ArgumentValidationException if the {@link AttachmentSupplier#getName()
     *                                     attachment name} uses special characters,
     *                                     like {@code ..}
     * @throws ServiceException            if the attachment cannot be stored
     */
    @Nonnull
    Attachment save(@Nonnull Repository repository, @Nonnull AttachmentSupplier attachmentSupplier);

    /**
     * Saves the metadata for an attachment that already exists. The ID that is used for the metadata matches the
     * ID for the attachment.
     *
     * @param repository   the repository to which the attachments belong
     * @param attachmentId the ID of the attachment
     * @param metadata the metadata that is to be associated to the attachment
     * @return the attachment metadata object that is saved
     * @throws NoSuchObjectException if the attachment cannot be found in the database
     */
    @Nonnull
    AttachmentMetadata saveMetadata(@Nonnull Repository repository, long attachmentId,
                                    @Nonnull Map<String, Object> metadata);
}
