package com.atlassian.bitbucket.validation;

import com.atlassian.bitbucket.validation.annotation.FieldEquals;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import java.lang.reflect.Field;
import java.util.HashSet;
import java.util.Set;

/**
 * Validate that multiple fields are identical.
 */
public class FieldEqualsValidator implements ConstraintValidator<FieldEquals, Object> {

    private String[] fields;
    private String message;
    private String reportingField;

    @Override
    public void initialize(FieldEquals constraint) {
        fields = constraint.fields();
        reportingField = constraint.reportingField();
        message = constraint.message();
    }

    @Override
    public boolean isValid(Object object, ConstraintValidatorContext context) {
        final boolean fieldMatches = fieldMatches(object, fields);
        if (!fieldMatches) {
            context.disableDefaultConstraintViolation();
            context.buildConstraintViolationWithTemplate(message).addNode(reportingField).addConstraintViolation();
        }
        return fieldMatches;
    }

    private boolean fieldMatches(Object value, String[] fields) {
        final Set<Object> values = new HashSet<>(fields.length);
        for (String fieldName : fields) {
            try {
                final Field field = findField(value.getClass(), fieldName);
                if (field == null) {
                    throw new IllegalStateException("Couldn't find field: " + fieldName);
                }

                if (!field.isAccessible()) {
                    field.setAccessible(true);
                }

                values.add(field.get(value));
            } catch (IllegalAccessException e) {
                throw new IllegalStateException("Inaccessible field: " + fieldName);
            }
        }

        return values.size() == 1;
    }

    // based on Spring's ReflectionUtils.findFields()
    private Field findField(Class<?> clazz, String name) {
        while (!Object.class.equals(clazz) && clazz != null) {
            final Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                if (name.equals(field.getName())) {
                    return field;
                }
            }
            clazz = clazz.getSuperclass();
        }

        return null;
    }

}
