package com.atlassian.bitbucket.util;

import com.google.common.collect.Iterables;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Objects;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Consumer;
import java.util.function.Function;

import static com.atlassian.bitbucket.util.MoreCollectors.toImmutableList;
import static com.atlassian.bitbucket.util.MoreStreams.streamIterable;

public class PageImpl<T> implements Page<T> {

    private final boolean lastPage;
    private final int nextPageStart;
    private final PageRequest pageRequest;
    private final int size;
    private final Iterable<T> values;

    /**
     * Creates a page from the provided {@code values} and explicitly sets {@link #getIsLastPage()} and
     * {@link #getSize() size}. The next {@link #getNextPageRequest() PageRequest} will be inferred from the
     * {@link PageRequest#getStart() start} and {@link PageRequest#getLimit() limit} of the provided
     * {@link PageRequest}.
     *
     * @param pageRequest the page request used to create this page
     * @param size the size of the page - i.e. the number of elements in this page
     * @param values an iterable of the values
     * @param lastPage if this is the last page
     */
    public PageImpl(PageRequest pageRequest, int size, Iterable<T> values, boolean lastPage) {
        this(pageRequest, values, lastPage, size, pageRequest.getStart() + pageRequest.getLimit());
    }

    /**
     * Creates a page from the provided {@code values} and explicitly sets {@link #getIsLastPage()} and
     * {@link #getSize() size}. The next {@link #getNextPageRequest() PageRequest} will start from the specified
     * {@code nextPageStart}, with the same {@link PageRequest#getLimit() limit} as the provided {@link PageRequest}.
     *
     * @param pageRequest the page request used to create this page
     * @param values an iterable of the values
     * @param lastPage if this is the last page
     * @param size the size of the page - i.e. the number of elements in this page
     * @param nextPageStart index of the element that will be the first in the next page
     * @since 4.12
     */
    public PageImpl(PageRequest pageRequest, Iterable<T> values, boolean lastPage, int size, int nextPageStart) {
        this.lastPage = lastPage;
        this.nextPageStart = nextPageStart;
        this.pageRequest = pageRequest;
        this.size = size;
        this.values = values;
    }

    /**
     * Creates a page from the provided {@code values} and explicitly sets {@link #getIsLastPage()}.
     *
     * @param pageRequest the page request used to create this page
     * @param values an iterable of the values
     * @param lastPage if this is the last page
     */
    public PageImpl(PageRequest pageRequest, Iterable<T> values, boolean lastPage) {
        this(pageRequest, Iterables.size(values), values, lastPage);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        PageImpl page = (PageImpl) o;
        return lastPage == page.lastPage &&
                size == page.size &&
                Objects.equals(pageRequest, page.pageRequest) &&
                Objects.equals(values, page.values);
    }

    /**
     * @param action the action to apply to each result
     * @since 6.3
     */
    @Override
    public void forEach(@Nonnull Consumer<? super T> action) {
        values.forEach(action);
    }

    @Override
    public boolean getIsLastPage() {
        return lastPage;
    }

    @Override
    public int getLimit() {
        return pageRequest.getLimit();
    }

    @Override
    public PageRequest getNextPageRequest() {
        return getIsLastPage() ? null : new PageRequestImpl(getNextPageOffset(), getLimit());
    }

    @Nonnull
    @Override
    public SortedMap<Integer, T> getOrdinalIndexedValues() {
        SortedMap<Integer, T> valueMap = new TreeMap<>();
        int index = getStart();
        for (T value : values) {
            valueMap.put(index++, value);
        }
        return valueMap;
    }

    @Override
    public int getSize() {
        return size;
    }

    @Override
    public int getStart() {
        return pageRequest.getStart();
    }

    @Nonnull
    @Override
    public Iterable<T> getValues() {
        return values;
    }

    @Override
    public int hashCode() {
        return Objects.hash(pageRequest, values, lastPage, size);
    }

    @Nonnull
    @Override
    public <E> PageImpl<E> transform(@Nonnull Function<? super T, ? extends E> transformFunction) {
        List<E> list = streamIterable(values)
                .map(transformFunction)
                .collect(toImmutableList());

        return new PageImpl<>(pageRequest, list, lastPage, size, nextPageStart);
    }

    protected int getNextPageOffset() {
        return nextPageStart;
    }
}
