package com.atlassian.bitbucket.user;

import javax.annotation.Nonnull;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * Handles rendering and parsing user mention strings in texts.
 *
 * @since 5.16
 */
public interface UserMentionScanner {

    /**
     * Returns all mentioned {@link ApplicationUser#getName() user names} in the text.
     * <p>
     * Note that the case of the user name returned matches the case used in the text and might not be identical to the
     * case of the {@link ApplicationUser#getName() ApplicationUser's name}.
     *
     * @param text text to search for user mentions
     * @param maxMentions maximum size of the returned set
     * @return set of {@link ApplicationUser#getName() user names} mentioned
     */
    @Nonnull
    Set<String> getMentionedUsers(@Nonnull CharSequence text, int maxMentions);

    /**
     * Calls <code>callback</code> for each user mention found in <code>text</code>.
     *
     * @param text text to search for user mentions
     * @param callback callback that gets called for every user mention found
     */
    void processMentions(@Nonnull CharSequence text, @Nonnull Consumer<UserMention> callback);

    /**
     * Renders the full user mention for a {@link ApplicationUser#getName() user name}.
     *
     * @param username {@link ApplicationUser#getName() user name} to render
     * @return rendered user mention
     */
    @Nonnull
    String renderMention(@Nonnull String username);

    /**
     * Replaces all user mentions with the string returned from the passed <code>replacementFunction</code>.
     * <p>
     * Return types for both the <code>replacementFunction</code> as well as this method are {@link Optional} so the calling
     * code can perform more efficient operations when no replacements have taken place.
     * <p>
     * Example usage:
     * <pre><code>
     * String replacedText = userMentionScanner.replaceMentions(text, mention -&gt;
     *         Optional.of(userMentionScanner.renderMention(mention.getUsername().replace("_", "-"))))
     *     .orElse(text);
     * </code></pre>
     * @param text text to search and replace user mentions in
     * @param replacementFunction function that optionally returns a replacement for the full mention of the user
     * @return the new text or {@link Optional#empty()} if no replacement took place
     */
    @Nonnull
    Optional<CharSequence> replaceMentions(@Nonnull CharSequence text, @Nonnull Function<UserMention, Optional<String>> replacementFunction);

    /**
     * Represents the location and content of a user mention in the text.
     */
    interface UserMention {

        /**
         * @return the end position of the user mention in the text
         */
        int getEnd();

        /**
         * @return the full mention string, including any meta or escape characters
         */
        @Nonnull
        String getFullMention();

        /**
         * @return the full mention string, including any meta characters, but not including escape characters
         */
        @Nonnull
        String getFullUnescapedMention();

        /**
         * @return the {@link ApplicationUser#getName() user name} of the mention string, not including any escape characters
         */
        @Nonnull
        String getUsername();

        /**
         * @return the start position of the user mention in the text
         */
        int getStart();
    }
}
