package com.atlassian.bitbucket.scm.bulk;

import com.atlassian.bitbucket.commit.BulkCommitsRequest;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.AbstractCommandParameters;
import com.google.common.collect.ImmutableSetMultimap;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.Set;

import static com.atlassian.bitbucket.commit.AbstractCommitsRequest.UNLIMITED_MESSAGE_LENGTH;
import static java.util.Objects.requireNonNull;

/**
 * @since 5.8
 */
public class BulkCommitsCommandParameters extends AbstractCommandParameters {

    private final Map<Repository, Set<String>> commits;
    private final boolean ignoringMissing;
    private final int maxMessageLength;
    private final boolean withMessages;

    @SuppressWarnings("unchecked")
    private BulkCommitsCommandParameters(Builder builder) {
        commits = (Map) builder.commits.build().asMap();
        ignoringMissing = builder.ignoringMissing;
        //If builder.withMessages is true, normalize the requested length. Otherwise, ignore it. This means the
        //length can either be 0 because it was set that way explicitly, or because withMessages was false
        maxMessageLength = builder.withMessages ? Math.max(UNLIMITED_MESSAGE_LENGTH, builder.maxMessageLength) : 0;
        //Only set withMessages to true if builder.withMessages was true _and_ the max length isn't 0
        withMessages = builder.withMessages && maxMessageLength != 0;
    }

    @Nonnull
    public Map<Repository, Set<String>> getCommits() {
        return commits;
    }

    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    public boolean isIgnoringMissing() {
        return ignoringMissing;
    }

    public boolean isWithMessages() {
        return withMessages;
    }

    public static class Builder {

        private final ImmutableSetMultimap.Builder<Repository, String> commits;

        private boolean ignoringMissing;
        private int maxMessageLength;
        private boolean withMessages;

        public Builder() {
            commits = ImmutableSetMultimap.builder();
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
            withMessages = true;
        }

        public Builder(@Nonnull BulkCommitsRequest request) {
            requireNonNull(request, "request");

            commits = ImmutableSetMultimap.builder();
            ignoringMissing = request.isIgnoringMissing();
            maxMessageLength = request.getMaxMessageLength().orElse(UNLIMITED_MESSAGE_LENGTH);
            withMessages = true;

            request.getCommits().forEach(commits::putAll);
        }

        @Nonnull
        public BulkCommitsCommandParameters build() {
            return new BulkCommitsCommandParameters(this);
        }

        @Nonnull
        public Builder commit(@Nonnull Repository repository, @Nonnull String value) {
            commits.put(requireNonNull(repository, "repository"), requireNonBlank(value, "commit"));

            return this;
        }

        @Nonnull
        public Builder commits(@Nonnull Repository repository, @Nonnull Iterable<String> values) {
            commits.putAll(requireNonNull(repository, "repository"), requireNonNull(values, "commits"));

            return this;
        }

        @Nonnull
        public Builder ignoreMissing(boolean value) {
            ignoringMissing = value;

            return this;
        }

        @Nonnull
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;

            return this;
        }

        @Nonnull
        public Builder withMessages(boolean value) {
            withMessages = value;

            return this;
        }
    }
}
