package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @since 5.0
 */
public class CommonAncestorCommandParameters extends AbstractCommandParameters {

    private final Set<String> commitIds;
    private final Repository secondaryRepository;

    private CommonAncestorCommandParameters(Builder builder) {
        commitIds = builder.commitIds.build();
        checkArgument(commitIds.size() > 0, "Must specify at least 1 commit ID");
        secondaryRepository = builder.secondaryRepository;
    }

    @Nonnull
    public Set<String> getCommitIds() {
        return commitIds;
    }

    @Nullable
    public Repository getSecondaryRepository() {
        return secondaryRepository;
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableSet.Builder<String> commitIds;

        private Repository secondaryRepository;

        public Builder() {
            commitIds = ImmutableSet.builder();
        }

        @Nonnull
        public CommonAncestorCommandParameters build() {
            return new CommonAncestorCommandParameters(this);
        }

        @Nonnull
        public Builder commitId(@Nullable String value, @Nullable String...moreValues) {
            addIf(Objects::nonNull, commitIds, value, moreValues);

            return this;
        }

        @Nonnull
        public Builder commitId(@Nullable Iterable<String> values) {
            addIf(Objects::nonNull, commitIds, values);

            return this;
        }

        @Nonnull
        public Builder secondaryRepository(@Nullable Repository value) {
            secondaryRepository = value;

            return this;
        }
    }
}
