package com.atlassian.bitbucket.request;

import com.atlassian.bitbucket.event.request.RequestEndedEvent;
import com.atlassian.bitbucket.event.request.RequestStartedEvent;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.function.Consumer;

/**
 * Sets up and tears down all logic for processing requests, such as logging, access logging, profiling and publishing
 * {@link RequestStartedEvent} and
 * {@link RequestEndedEvent}.
 */
public interface RequestManager {

    /**
     * Sets up and tears down all logic for processing requests.
     * @param callback called after the {@link RequestStartedEvent} has been published.
     * @param requestInfoProvider provides request details such as sessionId, remoteAddress that are used to
     *                            set up logging.
     * @param <T> type of the request
     * @param <E> type of exception
     * @return the return value of the callback
     * @throws E when {@link RequestCallback#withRequest(RequestContext)} throws an exception.
     */
    @Nullable
    <T, E extends Exception> T doAsRequest(@Nonnull RequestCallback<T, E> callback,
                                           @Nonnull RequestInfoProvider requestInfoProvider) throws E;

    /**
     * @return the context of the current request. Returns {@code null} if there is no current request, for instance
     *         when running a task in a background thread.
     */
    @Nullable
    RequestContext getRequestContext();

    /**
     * @return a highly unique (though not guaranteeably unique) request ID. Returns {@code null} if there is no
     *         current request, for instance when running a task in a background thread.
     * @see RequestContext#getId()
     * @since 5.0
     */
    @Nullable
    String getRequestId();

    /**
     * @return the metadata of the current request. Returns {@code null} if there is no current request.
     */
    @Nullable
    RequestMetadata getRequestMetadata();

    /**
     * Creates a new {@link RequestLocal} to get/set values in the <em>current</em> {@link RequestContext}.
     * Conceptually similar to a {@link ThreadLocal}, but tied to the current {@link RequestContext} instead of
     * the current thread.
     *
     * @param <T> the value type
     * @return the {@link RequestLocal}
     * @since 7.9
     */
    @Nonnull
    <T> RequestLocal<T> newRequestLocal();

    /**
     * Creates a new {@link RequestLocal} to get/set values in the <em>current</em> {@link RequestContext}.
     * Conceptually similar to a {@link ThreadLocal}, but tied to the current {@link RequestContext} instead of
     * the current thread.
     *
     * @param cleanupListener optional listener that is invoked when an {@link RequestContext} has completed and all
     *                        values in the context are removed. The {@code cleanupListener} is NOT called if the
     *                        {@link RequestLocal} does not contain a value for the {@link RequestContext} that is
     *                        being wrapped up.
     * @param <T> the value type
     * @return the {@link RequestLocal}
     * @since 7.9
     */
    @Nonnull
    <T> RequestLocal<T> newRequestLocal(@Nullable Consumer<T> cleanupListener);
}
